/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

/*
 *   assignment to out via operator, using calculate() method
 *
 *   Tests:
 *           single rate out / single rate in  (matrix,matrix,matrix,matrix,?in?)
 *                                             vector out / vector in
 *                                             port out   / vector in
 *                                             vector out / port in
 *                                             port out   / port in
 *
 *           single rate out / multi rate in  (matrix,matrix,matrix,matrix,?in?)
 *                                             vector out / matrix in
 *                                             port out   / matrix in
 *                                             vector out / port in
 *                                             port out   / port in
 *
 *           multi rate out / single rate in  (matrix,matrix,matrix,matrix, sc_core::sc_time,?in?)
 *           //delay is required due causality (sc_time argument)
 *                                             matrix out / vector in
 *                                             port out   / vector in
 *                                             matrix out / port in
 *                                             port out   / port in
 *
 *           multi rate out / multi rate in  (matrix,matrix,matrix,matrix,?in?)
 *                                             matrix out / matrix in
 *                                             port out   / matrix in
 *                                             matrix out / port in
 *                                             port out   / port in
 *
 *   Test of equation system without states (A, B and C have size zero)
 */


#include <systemc-ams>
#include "test_utilities.h"

#include <string>
#include <fstream>
#include <sstream>

///////////////////////////////////////////////////////////////////////////////

static const double DELAY_MAX_IN_SEC = 0.85e-3;
static const double DELAY_IN_SEC = 0.83e-3;

SCA_TDF_MODULE(ss_src)
{
  // single rate
  sca_tdf::sca_out<double> outp_const_1; // constant value (1.0)
  sca_tdf::sca_out<double> outp_ramp_1;  // ramp  y=t
  sca_tdf::sca_out<double> outp_t2_1;    // y=0.5*t*t
  sca_tdf::sca_out<double> outp_sin_1;
  sca_tdf::sca_out<double> outp_t3_1;    // y=0.5/3*t*t*t

  // multi rate
  sca_tdf::sca_out<double> outp_const_r;
  sca_tdf::sca_out<double> outp_ramp_r;
  sca_tdf::sca_out<double> outp_t2_r;
  sca_tdf::sca_out<double> outp_sin_r;
  sca_tdf::sca_out<double> outp_t3_r;

  // single rate
  sca_tdf::sca_out<sca_util::sca_vector<double> > outp_vec_1;

  // multi rate
  sca_tdf::sca_out<sca_util::sca_vector<double> > outp_vec_r;

  struct params
  {
    int rate;
    sca_core::sca_time sample_time;
    double f_sin;

    params()
    {
      rate = 5;
      sample_time = sca_core::sca_time(1.0, sc_core::SC_US);
      f_sin = 13.3e3;
    }
  };

  void set_attributes()
  {
    set_timestep(p.sample_time);

    outp_const_r.set_rate(p.rate);
    outp_ramp_r.set_rate(p.rate);
    outp_t2_r.set_rate(p.rate);
    outp_sin_r.set_rate(p.rate);
    outp_t3_r.set_rate(p.rate);
    outp_vec_r.set_rate(p.rate);
  }

  void initialize();
  void processing();

  ss_src( sc_core::sc_module_name nm, params pa = params() )
  : outp_const_1("outp_const_1"), outp_ramp_1("outp_ramp_1"), outp_t2_1("outp_t2_1"),
    outp_sin_1("outp_sin_1"), outp_t3_1("outp_t3_1"), outp_const_r("outp_const_r"),
    outp_ramp_r("outp_ramp_r"), outp_t2_r("outp_t2_r"), outp_sin_r("outp_sin_r"),
    outp_t3_r("outp_t3_r"), p(pa)
  {}

  params p;

private:
  sca_tdf::sca_ss  ss1, ss2, ss3, ss4;
  sca_util::sca_matrix<double> A1, B1, C1, D1;
};

void ss_src::initialize()
{
  A1(0, 0) = 0.0;
  A1(1, 0) = 1.0;  // ds = Integral(x)
  A1(1, 1) = 0.0;
  A1(2, 1) = 1.0;  // ds = integral(Integral(x))
  A1(2, 2) = 0.0;

  B1(0, 0) = 1.0;  // ds = x
  B1(1, 0) = 0.0;
  B1(1, 1) = 0.0;
  B1(2, 1) = 0.0;

  C1(0, 0) = 0.0;
  C1(1, 0) = 1.0;  // y1=t*x
  C1(2, 0) = 0.0;
  C1(2, 1) = 1.0;  // y2=0.5*t*t*x
  C1(3, 1) = 0.0;
  C1(4, 2) = 1.0;  // y2=0.5/3*t*t*t*x

  D1(0, 0) = 1.0; //  y0=x
  D1(2, 0) = 0.0;
  D1(3, 1) = 1.0; // y=sin(...)
  D1(4, 1) = 0.0;
}

// time domain implementation
void ss_src::processing()
{
  sca_util::sca_vector<double> inp;
  inp(0) = 1.0;
  inp(1) = std::sin(2.0 * M_PI * p.f_sin * get_time().to_seconds());

  // output to vector port sca_tdf::sca_out<sca_util::sca_vector<double> >
  // input vector sca_util::sca_vector<double>
  outp_vec_1 = ss1(A1, B1, C1, D1, inp); // single rate

  //output to vector port sca_tdf::sca_out<sca_util::sca_vector<double> >
  outp_vec_r = ss2(A1, B1, C1, D1, sc_core::sc_time(0.8, sc_core::SC_US), inp); // multi rate with 0.8us delay

  sca_util::sca_vector<double> out_vec;

  // output to vector sca_util::sca_vector<double>
  // input vector sca_util::sca_vector<double>
  out_vec = ss3(A1, B1, C1, D1, inp);

  outp_const_1 = out_vec(0);
  outp_ramp_1  = out_vec(1);
  outp_t2_1    = out_vec(2);
  outp_sin_1   = out_vec(3);
  outp_t3_1    = out_vec(4);

  sca_util::sca_matrix<double> out_mat(C1.n_rows(), p.rate);

  // output to matrix sca_util::sca_matrix<double>
  // input vector sca_util::sca_vector<double>
  out_mat = ss4(A1, B1, C1, D1, sc_core::sc_time(0.8, sc_core::SC_US), inp);

  for (long i = 0; i < p.rate; i++)
  {
    outp_const_r[i] = out_mat(0, i);
    outp_ramp_r[i]  = out_mat(1, i);
    outp_t2_r[i]    = out_mat(2, i);
    outp_sin_r[i]   = out_mat(3, i);
    outp_t3_r[i]    = out_mat(4, i);
  }
}

////////////////////////////////////////////////////////////////////////////////////////////////

SCA_TDF_MODULE (ss_test)
{
  // single rate
  sca_tdf::sca_in<double> in_const_1; // constant value (1.0)
  sca_tdf::sca_in<double> in_ramp_1;  // ramp  y=t
  sca_tdf::sca_in<double> in_t2_1;    // y=0.5*t*t
  sca_tdf::sca_in<double> in_sin_1;

  sca_tdf::sca_out<double> out_const_1;
  sca_tdf::sca_out<double> out_ramp_1;
  sca_tdf::sca_out<double> out_t2_1;
  sca_tdf::sca_out<double> out_sin_1;
  sca_tdf::sca_out<double> out_t3_1;

  sca_tdf::sca_out<double> out_const_3;
  sca_tdf::sca_out<double> out_ramp_3;
  sca_tdf::sca_out<double> out_t2_3;
  sca_tdf::sca_out<double> out_sin_3;
  sca_tdf::sca_out<double> out_t3_3;

  sca_tdf::sca_out<double> out_const_5;
  sca_tdf::sca_out<double> out_ramp_5;
  sca_tdf::sca_out<double> out_t2_5;
  sca_tdf::sca_out<double> out_sin_5;
  sca_tdf::sca_out<double> out_t3_5;

  sca_tdf::sca_out<double> out_const_7;
  sca_tdf::sca_out<double> out_ramp_7;
  sca_tdf::sca_out<double> out_t2_7;
  sca_tdf::sca_out<double> out_sin_7;
  sca_tdf::sca_out<double> out_t3_7;

  // multi rate
  sca_tdf::sca_in<double> in_const_r;
  sca_tdf::sca_in<double> in_ramp_r;
  sca_tdf::sca_in<double> in_t2_r;
  sca_tdf::sca_in<double> in_sin_r;

  // single rate
  sca_tdf::sca_in<sca_util::sca_vector<double> > in_vec_1;

  sca_tdf::sca_out<sca_util::sca_vector<double> > out_vec_2;
  sca_tdf::sca_out<sca_util::sca_vector<double> > out_vec_4;
  sca_tdf::sca_out<sca_util::sca_vector<double> > out_vec_6;
  sca_tdf::sca_out<sca_util::sca_vector<double> > out_vec_8;

  // multi rate
  sca_tdf::sca_in<sca_util::sca_vector<double> > in_vec_r;

  sca_tdf::sca_out<sca_util::sca_vector<double> > out_vec_r9;
  sca_tdf::sca_out<sca_util::sca_vector<double> > out_vec_r10;
  sca_tdf::sca_out<sca_util::sca_vector<double> > out_vec_r11;
  sca_tdf::sca_out<sca_util::sca_vector<double> > out_vec_r12;
  sca_tdf::sca_out<sca_util::sca_vector<double> > out_vec_r13;
  sca_tdf::sca_out<sca_util::sca_vector<double> > out_vec_r14;
  sca_tdf::sca_out<sca_util::sca_vector<double> > out_vec_r15;
  sca_tdf::sca_out<sca_util::sca_vector<double> > out_vec_r16;

  struct params // parameter
  {
    unsigned long rate, rate2;

    params() // default for parameter
    {
      rate = 5;
      rate2 = 3;
    }
  };

  void initialize();     // initialization
  void processing();     // time domain processing method
  void set_attributes();
  params p;

  // constructor
  ss_test(sc_core::sc_module_name nm, params pa = params())
  : in_const_1("in_const_1"), in_ramp_1("in_ramp_1"), 
    in_t2_1("in_t2_1"), in_sin_1("in_sin_1"), 
    out_const_1("out_const_1"), out_ramp_1("out_ramp_1"), 
    out_t2_1("out_t2_1"), out_sin_1("out_sin_1"), out_t3_1("out_t3_1"),
    out_const_3("out_const_3"), out_ramp_3("out_ramp_3"), 
    out_t2_3("out_t2_3"), out_sin_3("out_sin_3"), out_t3_3("out_t3_3"),
    out_const_5("out_const_5"), out_ramp_5("out_ramp_5"), 
    out_t2_5("out_t2_5"), out_sin_5("out_sin_5"), out_t3_5("out_t3_5"),
    out_const_7("out_const_7"), out_ramp_7("out_ramp_7"), 
    out_t2_7("out_t2_7"), out_sin_7("out_sin_7"), out_t3_7("out_t3_7"),
    in_const_r("in_const_r"), in_ramp_r("in_ramp_r"), in_t2_r("in_t2_r"),
    in_sin_r("in_sin_r"),
    in_vec_1("in_vec_1"), out_vec_2("out_vec_2"), out_vec_4("out_vec_4"),
    out_vec_6("out_vec_6"), out_vec_8("out_vec_8"),
    in_vec_r("in_vec_r"),
    out_vec_r9("out_vec_r9"), out_vec_r10("out_vec_r10"), out_vec_r11("out_vec_r11"),
    out_vec_r12("out_vec_r12"), out_vec_r13("out_vec_r13"), out_vec_r14("out_vec_r14"),
    out_vec_r15("out_vec_r15"), out_vec_r16("out_vec_r16"), p(pa)
  {}

  // definition of local variables
private:
  sca_tdf::sca_ss ss1, ss2, ss3, ss4, ss5, ss6, ss7, ss8, ss9, ss10, ss11, ss12, ss13, ss14, ss15, ss16;
  sca_tdf::sca_ss ss_delay;
  sca_util::sca_matrix<double> A, B, C, D;
  sca_util::sca_matrix<double> A_delay, B_delay, C_delay, D_delay;

  double cnt;
};

void ss_test::set_attributes()
{
  in_const_r.set_rate(p.rate);
  in_ramp_r.set_rate(p.rate);
  in_t2_r.set_rate(p.rate);
  in_sin_r.set_rate(p.rate);

  in_vec_r.set_rate(p.rate);

  out_vec_r9.set_rate(p.rate);
  out_vec_r10.set_rate(p.rate);
  out_vec_r11.set_rate(p.rate);
  out_vec_r12.set_rate(p.rate);

  out_vec_r13.set_rate(p.rate2);
  out_vec_r14.set_rate(p.rate2);
  out_vec_r15.set_rate(p.rate2);
  out_vec_r16.set_rate(p.rate2);

  ss1.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  ss2.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  ss3.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  ss4.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  ss5.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  ss6.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  ss7.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  ss8.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  ss9.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  ss10.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  ss11.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  ss12.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  ss13.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  ss14.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  ss15.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  ss16.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
}

void ss_test::initialize()
{
  A(0, 0) = 0.0;
  B(0, 4) = 1.0;  // ds=x

  C(4, 0) = 1.0;  // y= Integral(x)

  D(0, 0) = 1.0;
  D(1, 1) = 1.0;
  D(2, 2) = 1.0;
  D(3, 3) = 1.0;
  D(4, 4) = 0.0;

  D_delay(0, 0) = 1.0;
  D_delay(1, 1) = 1.0;
  D_delay(2, 2) = 1.0;
  D_delay(3, 3) = 1.0;
  D_delay(4, 4) = 1.0;
}

// time domain implementation
void ss_test::processing()
{
  sca_util::sca_vector<double> x;

  x(0) = in_const_1.read();
  x(1) = in_ramp_1.read();
  x(2) = in_t2_1.read();
  x(3) = in_sin_1.read();
  x(4) = in_t2_1.read();

  ////////// single rate in / single rate out //////////////

  // vector out / vector in
  const sca_util::sca_vector<double> y1 = 
    ss1.calculate(A, B, C, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), x);

  out_const_1 = y1(0);
  out_ramp_1  = y1(1);
  out_t2_1    = y1(2);
  out_sin_1   = y1(3);
  out_t3_1    = y1(4);

  // port out / vector in
  out_vec_2 = ss2.calculate(A, B, C, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), x);

  // vector out / port in
  const sca_util::sca_vector<double> y3 = 
    ss3.calculate(A, B, C, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), in_vec_1);

  out_const_3 = y3(0);
  out_ramp_3  = y3(1);
  out_t2_3    = y3(2);
  out_sin_3   = y3(3);
  out_t3_3    = y3(4);

  // port out / port in
  out_vec_4 = ss4.calculate(A, B, C, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), in_vec_1);

  ///////// multi rate in (rate=5) / single rate out ////////////////
  // -800  -600  -400  -200  0   time stamps (ps) of vector elements
  // this assignment corresponds to a delay of -0.8us (negative!!!)
  // assign time 0 -> -800, 200 -> -600, ...
  sca_util::sca_matrix<double> x_mat(D.n_rows(), p.rate);

  for (unsigned long i = 0; i < p.rate; i++)
  {
    x_mat(0, i) = in_const_r.read(i);
    x_mat(1, i) = in_ramp_r.read(i);
    x_mat(2, i) = in_t2_r.read(i);
    x_mat(3, i) = in_sin_r.read(i);
    x_mat(4, i) = in_t2_r.read();
  }

  sca_util::sca_matrix<double> x_mat_delay(D.n_rows(), p.rate);

  // remove negative delay of -0.8us
  // init init init init 0  | 200 400 600 800 1us | ...
  x_mat_delay = ss_delay(A_delay, B_delay, C_delay, D_delay,
                         (in_const_r.get_rate()-1) * in_const_r.get_timestep(), x_mat);

  // vector out / matrix in (rate=5)
  // use delayed matrix, due the last vector value corresponds to the out port value
  const sca_util::sca_vector<double> y5 = 
    ss5.calculate(A, B, C, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), x_mat_delay);

  out_const_5 = y5(0);
  out_ramp_5  = y5(1);
  out_t2_5    = y5(2);
  out_sin_5   = y5(3);
  out_t3_5    = y5(4);

  ////////////////////////////////////////////////////////////////////////

  // port out / matrix in (rate=5)
  // use delayed matrix, due the last vector value corresponds to the out port value
  out_vec_6 = ss6.calculate(A, B, C, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), x_mat_delay);

  // vector out / port in (rate=5)
  const sca_util::sca_vector<double> y7 = ss7.calculate(A, B, C, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), in_vec_r);

  out_const_7 = y7(0);
  out_ramp_7  = y7(1);
  out_t2_7    = y7(2);
  out_sin_7   = y7(3);
  out_t3_7    = y7(4);

  // port out (single rate) / port in (rate=5)
  out_vec_8 = ss8.calculate(A, B, C, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), in_vec_r);

  ////////// single rate in / multi rate out /////////////

  ////////// single rate in / multi rate out (rate=5) /////////////

  // matrix out / vector in
  sca_util::sca_matrix<double> y9(D.n_rows(), p.rate);
  y9 = ss9.calculate(A, B, C, D, sc_core::sc_time(DELAY_IN_SEC-0.8e-6, sc_core::SC_SEC), x);

  for (unsigned long i = 0; i < p.rate; i++)
  {
    sca_util::sca_vector<double> tmp;

    for (unsigned long j = 0; j < y9.n_rows(); j++)
    {
      tmp(j) = y9(j, i);
    }

    out_vec_r9[i] = tmp;
  }

  // overall delay 0.8us+0.83ms -0.8us = 0.83ms

  /////////////////////////////////////////////////////////////////////////

  // port out (rate=5) / vector in
  // 0.8us delay for causality
  out_vec_r10 = ss10.calculate(A, B, C, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), x);


  // overall delay 0.83 ms

  //////////////////////

  // matrix out (rate=5)  / port in (single rate)
  sca_util::sca_matrix<double> y11(D.n_rows(), p.rate);

  y11 = ss11.calculate(A, B, C, D, sc_core::sc_time(DELAY_IN_SEC-0.8e-6, sc_core::SC_SEC), in_vec_1);

  for (unsigned long i = 0; i < p.rate; i++)
  {
    sca_util::sca_vector<double> tmp;

    for (unsigned long j = 0; j < y11.n_rows(); j++)
    {
      tmp(j) = y11(j, i);
    }

    out_vec_r11[i] = tmp;
  }

  // overall delay 0.8us + 0.83ms -0.8us = 0.83ms

  //////////////////////////////////////////////////////////

  // port out (rate 5) / port in (single rate)
  // 0.8us delay for causality
  out_vec_r12 = ss12.calculate(A, B, C, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), in_vec_1);

  // overall delay 0.83ms

  ////////// multi rate in / multi rate out /////////////

  // matrix out (rate=3) / matrix in (rate=5)
  sca_util::sca_matrix<double> y13(D.n_rows(), p.rate2);

  // 0     1     2     3    4              sample index
  // 0    200   400   600  800             time of inport sample
  //-800 -600  -400  -200  0               time of vector sample
  // 0    200   400   600  800             time of delayed vector
  //    -666      -333     0               time of out vector
  //                       0   333  666    time of out port

  // we use the delayed matrix to remove the -0.8us (negative) delay from
  // the inport assignment

  y13 = ss13.calculate(A, B, C, D, sc_core::sc_time(DELAY_IN_SEC - 2.0e-6 / 3.0, sc_core::SC_SEC), x_mat_delay);

  // this assignment corresponds to a delay of 0.666us
  // -666 -> 0, -333 -> 333, 0 -> 666
  for (unsigned long i = 0; i < p.rate2; i++)
  {
    sca_util::sca_vector<double> tmp;

    for (unsigned long j = 0; j < y13.n_rows(); j++)
    {
      tmp(j) = y13(j, i);
    }

    out_vec_r13[i] = tmp;
  }

  // -> the overall delay will be 0.666us- 0.666us =0.83ms

  ///////////////////////////////////////////////////////////////////////////
  ///////////////////////////////////////////////////////////////////////////

  //  0     1     2     3    4              sample index
  //  0    200   400   600  800             time of inport sample
  // -800 -600  -400  -200  0               time of vector sample
  //                       0    333   666  time of outport sample

  // to achieve, that inport==outport we must delay the matrix by 800ps
  // port out (rate=3) / vector in (rate=5)
  // a delay is required due the matrix time ends at 0 and the port time at 1us/3*2

  // port out / matrix in
  out_vec_r14 = ss14.calculate(A, B, C, D, sc_core::sc_time(DELAY_IN_SEC + 0.8e-6, sc_core::SC_SEC), x_mat);

  ///////////////////////////////////////////////////////////

  // matrix out (rate=3) / port in (rate=5)
  sca_util::sca_matrix<double> y15(D.n_rows(), p.rate2);

  // inport time stamps:     0 200 400 600 800
  // out vector timestamps: -666 -333 0 (init init 0) | 333 666
  y15 = ss15.calculate(A, B, C, D, sc_core::sc_time(DELAY_IN_SEC - 2.0e-6 / 3.0, sc_core::SC_SEC), in_vec_r);

  for (unsigned long i = 0; i < p.rate2; i++)
  {
    sca_util::sca_vector<double> tmp;

    for (unsigned long j = 0; j < y15.n_rows(); j++)
    {
      tmp(j) = y15(j, i);
    }

    out_vec_r15.write(tmp, i);
  }

  //port out (rate=3) / port in (rate=5)
  out_vec_r16 = ss16.calculate(A, B, C, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), in_vec_r);
}

SCA_TDF_MODULE(vector_trace)
{
  sca_tdf::sca_in<sca_util::sca_vector<double> > in_vec;

  sca_tdf::sca_trace_variable<double> c0;
  sca_tdf::sca_trace_variable<double> c1;
  sca_tdf::sca_trace_variable<double> c2;
  sca_tdf::sca_trace_variable<double> c3;
  sca_tdf::sca_trace_variable<double> c4;

  void processing()
  {
    if (in_vec.read().length() != 5)
    {
      SC_REPORT_ERROR("vector_trace", "not supported vector length");
    }

    c0 = in_vec.read()(0);
    c1 = in_vec.read()(1);
    c2 = in_vec.read()(2);
    c3 = in_vec.read()(3);
    c4 = in_vec.read()(4);
  }

  vector_trace( sc_core::sc_module_name, sca_util::sca_trace_file* tf, std::string nm)
  {
    sca_util::sca_trace(tf, c0, nm + "_0");
    sca_util::sca_trace(tf, c1, nm + "_1");
    sca_util::sca_trace(tf, c2, nm + "_2");
    sca_util::sca_trace(tf, c3, nm + "_3");
    sca_util::sca_trace(tf, c4, nm + "_4");
  }
};

namespace sca_util
{
  void sca_trace(sca_util::sca_trace_file* tf, sca_tdf::sca_signal<sca_util::sca_vector<double> >& sig, std::string nm)
  {
    vector_trace* tfm = new vector_trace(sc_core::sc_gen_unique_name("vector_trace"), tf, nm);
    tfm->in_vec(sig);
  }
}

bool check_val(double val, double e_val, double rel_err, double abs_err)
{
  if ( (std::fabs(val - e_val) > abs_err) && (std::fabs(val - e_val) / e_val > rel_err))
  {
    std::cout << " Expect value: " << e_val << " read value: " << val << std::endl;
    return true;
  }
  return false;
}

int check_values(double time, double delay, double cval, double rampv, double t2_val, double sin_val,
                 double rel_err = 1e-10, double abs_err = 1e-10)
{
  double ctime = time-delay;
  double e_cval = 0.0;
  double e_rampv = 0.0;
  double e_t2_val = 0.0;
  double e_sin_val = 0.0;
  double e_t3_val = 0.0;

  if (ctime >= 0.0)
  {
    e_cval = 1.0;
    e_rampv = ctime;
    e_t2_val = 0.5 * ctime * ctime;
    e_sin_val = std::sin(2.0 * M_PI * 13.3e3 * ctime);
    e_t3_val = 0.5 / 3.0 * ctime * ctime * ctime;
  }
  else
  {
    return 0;
  }

  if (check_val(cval, e_cval, rel_err, abs_err))
  {
    return 1;
  }

  if (check_val(rampv, e_rampv, rel_err, abs_err))
  {
    return 2;
  }

  if (check_val(t2_val, e_t2_val, rel_err, abs_err))
  {
    return 3;
  }

  return 0;
}


bool check_file(std::string fname, const std::vector<double>& delays)
{
  std::ifstream fin;
  fin.open(fname.c_str());

  if (fin.fail())
  {
    std::ostringstream str;
    str << "Can't open file: " << fname;
    SC_REPORT_ERROR("SS_CHECK", str.str().c_str());
  }

  std::string line_in;
  unsigned long line_cnt = 0;
  if (!std::getline(fin, line_in))
  {
    std::ostringstream str;
    str << "Can't get header line from file: " << fname;
    SC_REPORT_ERROR("SS_CHECK", str.str().c_str());
  }
  line_cnt++;

  std::vector<std::string> names;
  std::istringstream nistr(line_in);

  std::string name;
  if (!(nistr >> name) && (name != "%time"))
  {
    std::ostringstream str;
    str << "Wrong header line in file: " << fname;
    SC_REPORT_ERROR("SS_CHECK", str.str().c_str());
  }

  while (nistr >> name)
  {
    names.push_back(name);
  }

  while (std::getline(fin, line_in))
  {
    std::istringstream istr(line_in);
    line_cnt++;

    double ctime;
    istr >> ctime;

    if (istr.fail())
    {
      std::ostringstream str;
      str << "Can't get time in file: " << fname << " line: " << line_cnt;
      SC_REPORT_ERROR("SS_CHECK", str.str().c_str());
    }

    for (unsigned long i = 0; i < names.size(); i += 5)
    {
      double cval, rval, t2val, sinval, t3val;

      istr >> cval;
      istr >> rval;
      istr >> t2val;
      istr >> sinval;
      istr >> t3val;

      if (istr.fail())
      {
        std::ostringstream str;
        str << "Failed to read values from file: " << fname << " line: " << line_cnt;
        str << " for: " << names[i] << " " << names[i+1] << " " << names[i+2]
            << " " << names[i+3] << " " << names[i+4];
        str << " at time: " << ctime;
        SC_REPORT_ERROR("SS_CHECK", str.str().c_str());
      }

      int err = check_values(ctime, delays[i / 5], cval, rval, t2val, sinval, t3val);
      if (err != 0)
      {
        std::ostringstream str;
        str << "Wrong value in file: " << fname << " line: " << line_cnt;
        str << " for signal: " << names[i+err-1] << " at time " << ctime;
        str << " delay: " << delays[i / 4];
        SC_REPORT_ERROR("SS_CHECK", str.str().c_str());
      }
    }
  }
  return false;
}

int sc_main(int argn, char* argc[])
{
  TEST_LABEL_START;

  // define non-conservative signals
  sca_tdf::sca_signal<double> s_out_const1, s_out_ramp1, s_out_t21, s_out_sin1, s_out_t31;
  sca_tdf::sca_signal<double> s_out_constr, s_out_rampr, s_out_t2r, s_out_sinr, s_out_t3r;

  sca_tdf::sca_signal<double> s_out_const_1, s_out_ramp_1, s_out_t2_1, s_out_sin_1, s_out_t3_1;
  sca_tdf::sca_signal<double> s_out_const_3, s_out_ramp_3, s_out_t2_3, s_out_sin_3, s_out_t3_3;
  sca_tdf::sca_signal<double> s_out_const_5, s_out_ramp_5, s_out_t2_5, s_out_sin_5, s_out_t3_5;
  sca_tdf::sca_signal<double> s_out_const_7, s_out_ramp_7, s_out_t2_7, s_out_sin_7, s_out_t3_7;

  sca_tdf::sca_signal<sca_util::sca_vector<double> > s_outvec_1, s_outvec_r;

  sca_tdf::sca_signal<sca_util::sca_vector<double> > s_outvec_2, s_outvec_4, s_outvec_6, s_outvec_8;

  sca_tdf::sca_signal<sca_util::sca_vector<double> > s_outvec_r9, s_outvec_r10, s_outvec_r11, s_outvec_r12;
  sca_tdf::sca_signal<sca_util::sca_vector<double> > s_outvec_r13, s_outvec_r14, s_outvec_r15, s_outvec_r16;

  // instantiate and connect components

  ss_src* i_src = new ss_src("i_src");
  i_src->outp_const_1(s_out_const1);
  i_src->outp_ramp_1(s_out_ramp1);
  i_src->outp_t2_1(s_out_t21);
  i_src->outp_sin_1(s_out_sin1);
  i_src->outp_t3_1(s_out_t31);

  i_src->outp_const_r(s_out_constr);
  i_src->outp_ramp_r(s_out_rampr);
  i_src->outp_t2_r(s_out_t2r);
  i_src->outp_sin_r(s_out_sinr);
  i_src->outp_t3_r(s_out_t3r);

  i_src->outp_vec_1(s_outvec_1);
  i_src->outp_vec_r(s_outvec_r);

  ss_test* i_ss_test = new ss_test("i_ss_test");
  i_ss_test->in_const_1(s_out_const1);
  i_ss_test->in_ramp_1(s_out_ramp1);
  i_ss_test->in_t2_1(s_out_t21);
  i_ss_test->in_sin_1(s_out_sin1);

  i_ss_test->in_const_r(s_out_constr);
  i_ss_test->in_ramp_r(s_out_rampr);
  i_ss_test->in_t2_r(s_out_t2r);
  i_ss_test->in_sin_r(s_out_sinr);

  i_ss_test->in_vec_1(s_outvec_1);
  i_ss_test->in_vec_r(s_outvec_r);

  i_ss_test->out_const_1(s_out_const_1);
  i_ss_test->out_ramp_1(s_out_ramp_1);
  i_ss_test->out_t2_1(s_out_t2_1);
  i_ss_test->out_sin_1(s_out_sin_1);
  i_ss_test->out_t3_1(s_out_t3_1);

  i_ss_test->out_vec_2(s_outvec_2);

  i_ss_test->out_const_3(s_out_const_3);
  i_ss_test->out_ramp_3(s_out_ramp_3);
  i_ss_test->out_t2_3(s_out_t2_3);
  i_ss_test->out_sin_3(s_out_sin_3);
  i_ss_test->out_t3_3(s_out_t3_3);

  i_ss_test->out_vec_4(s_outvec_4);

  i_ss_test->out_const_5(s_out_const_5);
  i_ss_test->out_ramp_5(s_out_ramp_5);
  i_ss_test->out_t2_5(s_out_t2_5);
  i_ss_test->out_sin_5(s_out_sin_5);
  i_ss_test->out_t3_5(s_out_t3_5);

  i_ss_test->out_vec_6(s_outvec_6);

  i_ss_test->out_const_7(s_out_const_7);
  i_ss_test->out_ramp_7(s_out_ramp_7);
  i_ss_test->out_t2_7(s_out_t2_7);
  i_ss_test->out_sin_7(s_out_sin_7);
  i_ss_test->out_t3_7(s_out_t3_7);

  i_ss_test->out_vec_8(s_outvec_8);
  i_ss_test->out_vec_r9(s_outvec_r9);
  i_ss_test->out_vec_r10(s_outvec_r10);
  i_ss_test->out_vec_r11(s_outvec_r11);
  i_ss_test->out_vec_r12(s_outvec_r12);

  i_ss_test->out_vec_r13(s_outvec_r13);
  i_ss_test->out_vec_r14(s_outvec_r14);
  i_ss_test->out_vec_r15(s_outvec_r15);
  i_ss_test->out_vec_r16(s_outvec_r16);

  /////// Tracing -> for each rate a different file

  sca_util::sca_trace_file* tf1 = sca_util::sca_create_tabular_trace_file("tdf_ss_1_1.dat");

  sca_util::sca_trace(tf1, s_out_const1, "s_out_const1");
  sca_util::sca_trace(tf1, s_out_ramp1, "s_out_ramp1");
  sca_util::sca_trace(tf1, s_out_t21, "s_out_t21");
  sca_util::sca_trace(tf1, s_out_sin1, "s_out_sin1");
  sca_util::sca_trace(tf1, s_out_t31, "s_out_t31");

  sca_util::sca_trace(tf1, s_outvec_1, "s_outvec_1");

  sca_util::sca_trace(tf1, s_out_const_1, "s_out_const_1");
  sca_util::sca_trace(tf1, s_out_ramp_1, "s_out_ramp_1");
  sca_util::sca_trace(tf1, s_out_t2_1, "s_out_t2_1");
  sca_util::sca_trace(tf1, s_out_sin_1, "s_out_sin_1");
  sca_util::sca_trace(tf1, s_out_t31, "s_out_t31");

  sca_util::sca_trace(tf1, s_out_const_3, "s_out_const_3");
  sca_util::sca_trace(tf1, s_out_ramp_3, "s_out_ramp_3");
  sca_util::sca_trace(tf1, s_out_t2_3, "s_out_t2_3");
  sca_util::sca_trace(tf1, s_out_sin_3, "s_out_sin_3");
  sca_util::sca_trace(tf1, s_out_t3_3, "s_out_t3_3");

  sca_util::sca_trace(tf1, s_out_const_5, "s_out_const_5");
  sca_util::sca_trace(tf1, s_out_ramp_5, "s_out_ramp_5");
  sca_util::sca_trace(tf1, s_out_t2_5, "s_out_t2_5");
  sca_util::sca_trace(tf1, s_out_sin_5, "s_out_sin_5");
  sca_util::sca_trace(tf1, s_out_t3_5, "s_out_t3_5");

  sca_util::sca_trace(tf1, s_out_const_7, "s_out_const_7");
  sca_util::sca_trace(tf1, s_out_ramp_7, "s_out_ramp_7");
  sca_util::sca_trace(tf1, s_out_t2_7, "s_out_t2_7");
  sca_util::sca_trace(tf1, s_out_sin_7, "s_out_sin_7");
  sca_util::sca_trace(tf1, s_out_t3_7, "s_out_t3_7");

  sca_util::sca_trace(tf1, s_outvec_2, "s_outvec_2");
  sca_util::sca_trace(tf1, s_outvec_4, "s_outvec_4");
  sca_util::sca_trace(tf1, s_outvec_6, "s_outvec_6");
  sca_util::sca_trace(tf1, s_outvec_8, "s_outvec_8");

  sca_util::sca_trace_file* tfr = sca_util::sca_create_tabular_trace_file("tdf_ss_1_rate.dat");

  sca_util::sca_trace(tfr, s_out_constr, "s_out_constr");
  sca_util::sca_trace(tfr, s_out_rampr, "s_out_rampr");
  sca_util::sca_trace(tfr, s_out_t2r, "s_out_t2r");
  sca_util::sca_trace(tfr, s_out_sinr, "s_out_sinr");
  sca_util::sca_trace(tfr, s_out_t3r, "s_out_t3r");

  sca_util::sca_trace(tfr, s_outvec_r, "s_outvec_r");

  sca_util::sca_trace(tfr, s_outvec_r9, "s_outvec_r9");
  sca_util::sca_trace(tfr, s_outvec_r10, "s_outvec_r10");
  sca_util::sca_trace(tfr, s_outvec_r11, "s_outvec_r11");
  sca_util::sca_trace(tfr, s_outvec_r12, "s_outvec_r12");

  sca_util::sca_trace_file* tfr2 = sca_util::sca_create_tabular_trace_file("tdf_ss_1_rate2.dat");

  sca_util::sca_trace(tfr2, s_outvec_r13, "s_outvec_r13");
  sca_util::sca_trace(tfr2, s_outvec_r14, "s_outvec_r14");
  sca_util::sca_trace(tfr2, s_outvec_r15, "s_outvec_r15");
  sca_util::sca_trace(tfr2, s_outvec_r16, "s_outvec_r16");

  ////////////////////////////////////////////////

  sc_core::sc_start(2.0, sc_core::SC_MS); //start time domain simulation for 5ms

  sca_util::sca_close_tabular_trace_file(tf1);
  sca_util::sca_close_tabular_trace_file(tfr);
  sca_util::sca_close_tabular_trace_file(tfr2);

  std::vector<double> delays;
  delays.resize(20, 0.0);

  for (unsigned long i = 2; i < 10; i++)
  {
    delays[i] = DELAY_IN_SEC;
  }

  // delay is not shown, due we propagate the first value to zero -> thus at time
  // zero the constant is valid -> the ramp starts at zero (begin of integration with 1)
  check_file("tdf_ss_1_1.dat", delays);

  delays.clear();
  delays.resize(20, 0.0);

  for (unsigned long i = 2; i < 6; i++)
  {
    delays[i] = DELAY_IN_SEC;
  }

  check_file("tdf_ss_1_rate.dat", delays);

  delays.resize(0);
  delays.resize(20, 0.0);

  for (unsigned long i = 0; i < 5; i++)
  {
    delays[i] = DELAY_IN_SEC;
  }

  check_file("tdf_ss_1_rate2.dat", delays);

  if (sc_core::sc_report_handler::get_count(sc_core::SC_ERROR) <= 0)
  {
    SC_REPORT_INFO("SS_DELAY_CHECK", "Test was successful");
  }

  TEST_LABEL_END;

  delete i_src, i_ss_test;

  return 0;
}
