/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

//  test02.cpp  - eln regression test
//  test with two tdf-resistors and others

#include <systemc-ams>
#include "test_utilities.h"

SC_MODULE(test02)
{
  // port declaration
  sca_tdf::sca_in<double> inp1;
  sca_tdf::sca_in<double> inp2;
  sca_tdf::sca_in<double> inp3;

  // node declaration
  sca_eln::sca_node n3;
  sca_eln::sca_node n4;
  sca_eln::sca_node_ref n5;

  // component declaration
  sca_eln::sca_tdf_vsource   *vin;
  sca_eln::sca_tdf_r         *tdf2r, *tdf2r2;

  struct params // parameter
  {

    params() // default for parameter
    {
    }
  };

  test02( sc_core::sc_module_name nm, params pa = params() )
  : inp1("inp1"), inp2("inp2"), inp3("inp3"),
    n3("n3"), n4("n4"), n5("n5"),
    p(pa)
  {
    architecture(); // generate netlist
  }

  ~test02();

 private:
  params p;
  void architecture(); // method for netlist
};

// netlist implementation

void test02::architecture()
{
  tdf2r2 = new sca_eln::sca_tdf_r("tdf2r2");
  tdf2r2->scale = 1.0;
  tdf2r2->n(n4);
  tdf2r2->p(n3);
  tdf2r2->inp(inp3);

  tdf2r = new sca_eln::sca_tdf_r("tdf2r");
  tdf2r->scale = 1.0e-1;
  tdf2r->n(n3);
  tdf2r->p(n5);
  tdf2r->inp(inp2);

  vin = new sca_eln::sca_tdf_vsource("vin");
  vin->inp(inp1);
  vin->p(n5);
  vin->n(n4);
}

test02::~test02()
{
  tdf2r2, tdf2r, vin;
}

void check_td_results(std::string fname)
{
  bool test_passed_with_success = true;
  std::ifstream tdf(fname.c_str());
  std::string line;
  long test_cnt = 0;

  getline(tdf, line);
  std::cout << "1..35" << std::endl;
  while (getline(tdf, line))
  {
    std::istringstream str(line);

    double time;
    str >> time;

    double node3, node4, node5, i_vin, i4, i5;
    str >> node3 >> node4 >> node5 >> i_vin >> i4 >> i5;

    if ( !((test_util::check_value(i4, 10.0) && (time <= 0.03)) ||
        (test_util::check_value(i4, 1e-13) && (time > 0.03))) )
    {
      std::cerr << "Error TD-check failed for i4 at: " << time << std::endl;
      std::cout << "not ";
      test_passed_with_success = false;
    }
    std::cout << "ok " << ++test_cnt << std::endl;

    if ( !((test_util::check_value(i5, 10.0) && (time <= 0.03)) ||
        (test_util::check_value(i5, 1e-13) && (time > 0.03))) )
    {
      std::cerr << "Error TD-check failed for i5 at: " << time << std::endl;
      std::cout << "not ";
      test_passed_with_success = false;
    }
    std::cout << "ok " << ++test_cnt << std::endl;

    if ( !(( test_util::check_value(i5, i4) && (time <= 0.03) ) ||
        (test_util::check_value(i5, i4, 1.4e-18) && (time > 0.03))) )
    {
      std::cerr << "Error TD-check failed for verification of currents at: " << time << std::endl;
      std::cout << "not ";
      test_passed_with_success = false;
    }
    std::cout << "ok " << ++test_cnt << std::endl;

    if ( !(( test_util::check_value(i5, -i_vin) && (time <= 0.03) ) ||
        (test_util::check_value(i5, -i_vin, 1.4e-18) && (time > 0.03))) )
    {
      std::cerr << "Error TD-check failed for verification of currents at: " << time << std::endl;
      std::cout << "not ";
      test_passed_with_success = false;
    }
    std::cout << "ok " << ++test_cnt << std::endl;

    if ( !(( test_util::check_value(-i_vin, i4) && (time <= 0.03) ) ||
        (test_util::check_value(-i_vin, i4, 1.4e-18) && (time > 0.03))) )
    {
      std::cerr << "Error TD-check failed for verification of currents at: " << time << std::endl;
      std::cout << "not ";
      test_passed_with_success = false;
    }
    std::cout << "ok " << ++test_cnt << std::endl;
  };

  if (test_passed_with_success) std::cout << "results ok." << std::endl;
}

int sc_main(int argc, char* argv[])
{
  TEST_LABEL_START;

  // define non-conservative signals
  sca_tdf::sca_signal<double> signal1;
  sca_tdf::sca_signal<double> signal2;
  sca_tdf::sca_signal<double> signal3;

  test_util::const_src::params src_params;
  src_params.ampl = 10;

  test_util::const_src* src = new test_util::const_src("source", src_params);
  src->outp(signal1);

  test_util::pulse_src2* swch = new test_util::pulse_src2("sca_sdf2r");
  swch->outp(signal2);

  test_util::pulse_src2::params src_params2;
  src_params2.val = 1.0e-3;

  test_util::pulse_src2* swch2 = new test_util::pulse_src2("sca_sdf2r2");
  swch2->outp(signal3);

  test02* cir = new test02 ("circuit");
  cir->inp1(signal3);
  cir->inp2(signal2);
  cir->inp3(signal1);

  sca_util::sca_trace_file* tf = sca_util::sca_create_tabular_trace_file("output4.dat");
  sca_util::sca_trace(tf, cir->n3, "node3");
  sca_util::sca_trace(tf, cir->n4, "node4");
  sca_util::sca_trace(tf, cir->n5, "node5");
  sca_util::sca_trace(tf, cir->vin, "i_through_vin");
  sca_util::sca_trace(tf, cir->tdf2r2, "i4");
  sca_util::sca_trace(tf, cir->tdf2r, "i5");

  sc_core::sc_start(80.0, sc_core::SC_MS);

  sca_util::sca_close_tabular_trace_file(tf);

  check_td_results("output4.dat");

  TEST_LABEL_END;

  delete src, swch, swch2, cir;

  return 0;
}
