/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

// test change rate for SystemC-AMS 2.0 dtdf

#include <systemc-ams>
#include "test_utilities.h"

test_util::sorted_stream sstr;

static std::vector<sc_core::sc_time> g_out_time_ref;

void check_trace_file(std::string name)
{
  std::fstream fin;

  fin.open(name.c_str());

  if (fin.fail())
  {
    std::ostringstream str;
    str << "Can't open trace file: " << name;
    SC_REPORT_ERROR("test", str.str().c_str());
    return;
  }

  std::string header;
  std::getline(fin, header);

  long cnt = 0;

  while (!fin.eof())
  {
    std::string line;

    std::getline(fin, line);
    std::istringstream line_str(line);

    double time;
    long   value;

    line_str >> time;
    if (line_str.fail()) continue; // ignore empty lines

    // stop at end time, due last value in trace may hold
    if (time + sc_core::sc_get_time_resolution().to_seconds()
         >= sc_core::sc_time_stamp().to_seconds()) break;

    line_str >> value;

    if ( (std::fabs(time - g_out_time_ref[cnt].to_seconds()) > 1e-12) &&
         (std::fabs(1.0 - time / g_out_time_ref[cnt].to_seconds()) > 1e-5))
    {
      std::ostringstream str;
      str << "Unexpexted time in trace file: " << name << " at time: " << time;
      str << " expected time: " << g_out_time_ref[cnt].to_seconds() << " cnt: " << cnt;
      SC_REPORT_ERROR("test", str.str().c_str());
      return;
    }

    if (value != cnt)
    {
      std::ostringstream str;
      str << "Unexpexted value in trace file: " << name << " at time: " << time;
      str << " expected value: " << cnt << " read value: " << value;
      SC_REPORT_ERROR("test", str.str().c_str());
      return;
    }
    cnt++;
  }
}

///////////////////////////////////////////////////////////////////

bool is_time_not_equal(const sca_core::sca_time& t1, const sca_core::sca_time& t2)
{
  sca_core::sca_time tdiff;
  if (t1 > t2) tdiff = t1 - t2;
  else         tdiff = t2 - t1;
  return tdiff > sc_core::sc_get_time_resolution();
}

SCA_TDF_MODULE( mod_a )
{
  sca_tdf::sca_out<double> outp;

  void print_status()
  {
    std::cout << "   are_attribute_changes_allowed: " << (are_attribute_changes_allowed() ? "yes" : "no") << std::endl;
    std::cout << "   is_dynamic:                    " << (is_dynamic() ? "yes" : "no") << std::endl;
    std::cout << "   are_attributes_changed:        " << (are_attributes_changed() ? "yes" : "no") << std::endl;
    std::cout << "   is_timestep_changed:           " << (is_timestep_changed() ? "yes" : "no") << std::endl;
    std::cout << "   get_timestep:                  " << get_timestep() << std::endl;
    std::cout << "   get_last_timestep:             " << get_last_timestep() << std::endl;
    std::cout << "   get_max_timestep:              ";
    if (get_max_timestep() == sca_core::sca_max_time()) std::cout << "sca_core::sca_max_time" << std::endl;
    else                                                std::cout << get_max_timestep() << std::endl;

    std::cout << "   get_time:                      " << get_time() << std::endl;
    std::cout << std::endl;
    std::cout << "   outp.get_delay()               " << outp.get_delay() << std::endl;
    std::cout << "   outp.get_rate()                " << outp.get_rate() << std::endl;
    std::cout << "   outp.get_time()                " << outp.get_time() << std::endl;
    for (unsigned long i = 0; i < outp.get_rate(); i++)
    {
      std::cout << "   outp.get_time(" << i << ")               " << outp.get_time(i) << std::endl;
    }

    std::cout << "   outp.get_timestep()            " << outp.get_timestep() << std::endl;
    for (unsigned long i = 0; i < outp.get_rate(); i++)
    {
      std::cout << "   outp.get_timestep(" << i << ")           " << outp.get_timestep(i) << std::endl;
    }

    std::cout << "   outp.get_last_timestep()       " << outp.get_last_timestep() << std::endl;
    for (unsigned long i = 0; i <outp.get_rate(); i++)
    {
      std::cout << "   outp.get_last_timestep(" << i << ")      " << outp.get_last_timestep(i) << std::endl;
    }

    std::cout << "   outp.get_max_timestep()        ";
    if (outp.get_max_timestep() == sca_core::sca_max_time()) std::cout << "sca_core::sca_max_time" << std::endl;
    else                                                     std::cout << outp.get_max_timestep() << std::endl;

    std::cout << "   outp.is_timestep_changed()     " << (outp.is_timestep_changed() ? "yes" : "no") << std::endl;
    for (unsigned long i = 0; i < outp.get_rate(); i++)
    {
      std::cout << "   outp.is_timestep_changed(" << i << ")    " << (outp.is_timestep_changed(i) ? "yes" : "no") << std::endl;
    }

    std::cout << "   outp.is_rate_changed()         " << (outp.is_rate_changed() ? "yes" : "no") << std::endl;
    std::cout << "   outp.is_delay_changed()        " << (outp.is_delay_changed() ? "yes" : "no") << std::endl;
  }

  void set_attributes()
  {
    outp.set_rate(3);
    outp.set_delay(5);
    sstr.redirect_cout(this->get_time(), 01);

    std::cout << "set_attributes() mod_a" << std::endl;

    set_timestep(1.0, sc_core::SC_MS);

    std::cout << "      outp.set_rate(3);" << std::endl;
    std::cout << "      outp.set_delay(5);" << std::endl;
    std::cout << "      set_timestep(1.0,sc_core::SC_MS);\n" << std::endl;

    sstr.restore_cout();
  }

  void initialize()
  {
    sstr.redirect_cout(this->get_time(), 11);

    std::cout << "--------------------------------------------" << std::endl;
    std::cout << get_time() << " initialize() mod_a" << std::endl;
    print_status();

    sstr.restore_cout();

    out_cnt = 0;

    for (unsigned long i = 0; i < outp.get_delay(); i++)
    {
      outp.initialize(out_cnt, i);
      out_cnt++;
    }

    last_time_step = this->get_timestep();
    last_time = this->get_time() - this->get_timestep(); //for the first timestep the propagated value is returned
    last_port_time = outp.get_time() - outp.get_timestep();
    last_port_timestep = outp.get_timestep();
  }

  void reinitialize()
  {
    sstr.redirect_cout(this->get_time(), 21);

    std::cout << "--------------------------------------------" << std::endl;
    std::cout << "--------------------------------------------" << std::endl;
    std::cout << get_time() << " reinitialize() mod_a" << std::endl;
    print_status();

    sstr.restore_cout();
  }

  void processing()
  {
    sstr.redirect_cout(this->get_time(), 31);

    std::cout << "--------------------------------------------" << std::endl;
    std::cout << get_time() << " processing() mod_a" << std::endl;
    print_status();

    sstr.restore_cout();

    for (unsigned long i = 0; i < outp.get_rate(); ++i)
    {
      outp.write(double(out_cnt), i);
      g_out_time_ref.push_back(outp.get_time(i));
      out_cnt++;
    }

    //// Consistency checks ///////////////////////////

    for (unsigned long i = 0; i < outp.get_rate(); ++i)
    {
      if (is_time_not_equal(outp.get_time(i) - last_port_time, outp.get_timestep(i)))
      {
        std::ostringstream str;
        str << "Wrong port timestep expect: " << outp.get_time(i) - last_port_time;
        str << " last_time: " << last_port_time << " current time: " << outp.get_time(i);
        str << " get_timestep: " << outp.get_timestep(i);
        str << " for sample: " << i;
        SC_REPORT_ERROR("Check", str.str().c_str());

      }
      last_port_time = outp.get_time(i);

      if (is_time_not_equal(last_port_timestep, outp.get_last_timestep(i)))
      {
        std::ostringstream str;
        str << "Wrong port last timestep expect: " << last_port_timestep;
        str << " outp.get_last_timestep: " << outp.get_last_timestep(i);
        str << " for sample: " << i;
        SC_REPORT_ERROR("Check", str.str().c_str());
      }
      last_port_timestep = outp.get_timestep(i);
    }

    if (is_time_not_equal(this->get_time() - last_time, this->get_timestep()))
    {
      std::ostringstream str;
      str << "Wrong timestep expect: " << this->get_time() - last_time;
      str << " last_time: " << last_time << " current time: " << this->get_time();
      str << " get_timestep: " << this->get_timestep();
      SC_REPORT_ERROR("Check", str.str().c_str());

    }
    last_time = this->get_time();


    if (is_time_not_equal(last_time_step, this->get_last_timestep()))
    {
      std::ostringstream str;
      str << "Wrong last timestep expect: " << last_time_step;
      str << " get_last_timestep: " << this->get_last_timestep();
      SC_REPORT_ERROR("Check", str.str().c_str());
    }
    last_time_step = this->get_timestep();
  }

  void change_attributes()
  {
    cnt++;

    sstr.redirect_cout(this->get_time(), 41);

    std::cout << std::endl << std::endl;
    std::cout << "+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++" << std::endl;
    std::cout << std::endl;
    std::cout << "--------------------------------------------" << std::endl;
    std::cout << get_time() << " change_attributes() mod_a  cnt: " << cnt << std::endl;

    switch(cnt)
    {
      case 1:
        break;
      case 2:
        // the order has no influence, due consistency check is after
        // change_attribute execution
        does_attribute_changes();
        std::cout << "--------------------------------------------" << std::endl;
        std::cout << "    does_attribute_changes();" << std::endl;
        break;
      case 3:
        does_no_attribute_changes();
        outp.set_rate(1);
        std::cout << "--------------------------------------------" << std::endl;
        std::cout << "    does_no_attribute_changes();" << std::endl;
        std::cout << "    outp.set_rate(1);" << std::endl;
        break;
      case 4:
        does_attribute_changes();
        std::cout << "--------------------------------------------" << std::endl;
        std::cout << "    does_attribute_changes();" << std::endl;
        break;
      case 5:
        request_next_activation(0.5, sc_core::SC_MS);
        std::cout << "--------------------------------------------" << std::endl;
        std::cout << "    request_next_activation(0.5,sc_core::SC_MS);" << std::endl;
        break;
      case 6:
        does_attribute_changes();
        outp.set_rate(5);
        std::cout << "--------------------------------------------" << std::endl;
        std::cout << "    does_attribute_changes();" << std::endl;
        std::cout << "    outp.set_rate(5);"         << std::endl;
        break;
      case 7:
        does_attribute_changes();
        outp.set_rate(2);
        std::cout << "--------------------------------------------" << std::endl;
        std::cout << "    does_attribute_changes();" << std::endl;
        std::cout << "    outp.set_rate(2);" << std::endl;
        break;
      case 8:
        does_attribute_changes();
        outp.set_rate(3);
        std::cout << "--------------------------------------------" << std::endl;
        std::cout << "    does_attribute_changes();" << std::endl;
        std::cout << "    outp.set_rate(3);" << std::endl;
        break;
      default:
        sc_core::sc_stop();
        break;
    }
    print_status();
    sstr.restore_cout();
  }

  SCA_CTOR( mod_a ) : outp("outp"), cnt(0), out_cnt(0)
  {}

 private:
  long cnt;
  long out_cnt;

  sca_core::sca_time last_time;
  sca_core::sca_time last_time_step;
  sca_core::sca_time last_port_time;
  sca_core::sca_time last_port_timestep;
};

////////////////////////////////////////////////////////////////////////////

SCA_TDF_MODULE( mod_b )
{
  sca_tdf::sca_in<double> inp;
  sca_tdf::sca_de::sca_out<long> out_sc;

  void print_status()
  {
    std::cout << std::setw(50)<< "" << "   are_attribute_changes_allowed: " << (are_attribute_changes_allowed() ? "yes" : "no") << std::endl;
    std::cout << std::setw(50)<< "" << "   is_dynamic:                    " << (is_dynamic() ? "yes" : "no") << std::endl;
    std::cout << std::setw(50)<< "" << "   are_attributes_changed:        " << (are_attributes_changed() ? "yes" : "no") << std::endl;
    std::cout << std::setw(50)<< "" << "   is_timestep_changed:           " << (is_timestep_changed() ? "yes" : "no") << std::endl;
    std::cout << std::setw(50)<< "" << "   get_timestep:                  " << get_timestep() << std::endl;
    std::cout << std::setw(50)<< "" << "   get_last_timestep:             " << get_last_timestep() << std::endl;
    std::cout << std::setw(50)<< "" << "   get_max_timestep:              ";
    if (get_max_timestep() == sca_core::sca_max_time()) std::cout << "sca_core::sca_max_time" << std::endl;
    else                                               std::cout << get_max_timestep() << std::endl;

    std::cout << std::setw(50)<< "" << "   get_time:                      " << get_time() << std::endl;
    std::cout << std::endl;
    std::cout << std::setw(50)<< "" << "   inp.get_delay()                " << inp.get_delay() << std::endl;
    std::cout << std::setw(50)<< "" << "   inp.get_rate()                 " << inp.get_rate() << std::endl;
    std::cout << std::setw(50)<< "" << "   inp.get_time()                 " << inp.get_time() << std::endl;
    std::cout << std::setw(50)<< "" << "   inp.get_time(1)                " << inp.get_time(1) << std::endl;
    std::cout << std::setw(50)<< "" << "   inp.get_timestep()             " << inp.get_timestep() << std::endl;
    std::cout << std::setw(50)<< "" << "   inp.get_timestep(1)            " << inp.get_timestep(1) << std::endl;
    std::cout << std::setw(50)<< "" << "   inp.get_last_timestep()        " << inp.get_last_timestep() << std::endl;
    std::cout << std::setw(50)<< "" << "   inp.get_last_timestep(1)       " << inp.get_last_timestep(1) << std::endl;
    std::cout << std::setw(50)<< "" << "   inp.get_max_timestep()         ";
    if (inp.get_max_timestep() == sca_core::sca_max_time()) std::cout << "sca_core::sca_max_time" << std::endl;
    else                                               std::cout << inp.get_max_timestep() << std::endl;

    std::cout << std::setw(50)<< "" << "   inp.is_timestep_changed()      " << (inp.is_timestep_changed() ? "yes" : "no") << std::endl;
    std::cout << std::setw(50)<< "" << "   inp.is_timestep_changed(1)     " << (inp.is_timestep_changed(1) ? "yes" : "no") << std::endl;
    std::cout << std::setw(50)<< "" << "   inp.is_rate_changed()          " << (inp.is_rate_changed() ? "yes" : "no") << std::endl;
    std::cout << std::setw(50)<< "" << "   inp.is_delay_changed()         " << (inp.is_delay_changed() ? "yes" : "no") << std::endl;

    std::cout << std::endl;
    std::cout << std::setw(50)<< "" << "   out_sc.get_delay()                " << out_sc.get_delay() << std::endl;
    std::cout << std::setw(50)<< "" << "   out_sc.get_rate()                 " << out_sc.get_rate() << std::endl;
    std::cout << std::setw(50)<< "" << "   out_sc.get_time()                 " << out_sc.get_time() << std::endl;
    std::cout << std::setw(50)<< "" << "   out_sc.get_time(1)                " << out_sc.get_time(1) << std::endl;
    std::cout << std::setw(50)<< "" << "   out_sc.get_timestep()             " << out_sc.get_timestep() << std::endl;
    std::cout << std::setw(50)<< "" << "   out_sc.get_timestep(1)            " << out_sc.get_timestep(1) << std::endl;
    std::cout << std::setw(50)<< "" << "   out_sc.get_last_timestep()        " << out_sc.get_last_timestep() << std::endl;
    std::cout << std::setw(50)<< "" << "   out_sc.get_last_timestep(1)       " << out_sc.get_last_timestep(1) << std::endl;
    std::cout << std::setw(50)<< "" << "   out_sc.get_max_timestep()         ";
    if (out_sc.get_max_timestep() == sca_core::sca_max_time()) std::cout << "sca_core::sca_max_time" << std::endl;
    else                                               std::cout << out_sc.get_max_timestep() << std::endl;

    std::cout << std::setw(50)<< "" << "   out_sc.is_timestep_changed()      " << (out_sc.is_timestep_changed() ? "yes" : "no") << std::endl;
    std::cout << std::setw(50)<< "" << "   out_sc.is_timestep_changed(1)     " << (out_sc.is_timestep_changed(1) ? "yes" : "no") << std::endl;
    std::cout << std::setw(50)<< "" << "   out_sc.is_rate_changed()          " << (out_sc.is_rate_changed() ? "yes" : "no") << std::endl;
    std::cout << std::setw(50)<< "" << "   out_sc.is_delay_changed()         " << (out_sc.is_delay_changed() ? "yes" : "no") << std::endl;
  }

  void set_attributes()
  {
    inp.set_rate(2);
    inp.set_delay(6);
    out_sc.set_rate(2);

    sstr.redirect_cout(this->get_time(), 02);

    std::cout << std::setw(50) << "" << "--------------------------------------------" << std::endl;
    std::cout << std::setw(50) << "" << " set_attributes() mod_b" << std::endl;
    std::cout << std::setw(50) << "" << "     inp.set_rate(2);" << std::endl;
    std::cout << std::setw(50) << "" << "     inp.set_delay(6);" << std::endl;
    std::cout << std::setw(50) << "" << "     out_sc.set_rate(2);" << std::endl;

    sstr.restore_cout();
  }

  void initialize()
  {
    sstr.redirect_cout(this->get_time(), 12);

    std::cout << std::setw(50) << "" <<"--------------------------------------------" << std::endl;
    std::cout << std::setw(50) << "" << get_time() << " initialize() mod_b" << std::endl;
    print_status();

    sstr.restore_cout();

    in_cnt = -inp.get_delay();

    for (unsigned long i = 0; i < inp.get_delay(); i++)
    {
      inp.initialize(in_cnt, i);
      in_cnt++;
    }

    in_cnt = -inp.get_delay();
    last_time_step = this->get_timestep();
    last_time = this->get_time() - this->get_timestep(); // for the first timestep the propagated value is returned
    last_port_time = inp.get_time() - inp.get_timestep();
    last_port_timestep = inp.get_timestep();
  }

  void reinitialize()
  {
    sstr.redirect_cout(this->get_time(), 22);

    std::cout << std::setw(50) << "" << "--------------------------------------------" << std::endl;
    std::cout << std::setw(50) << "" << "--------------------------------------------" << std::endl;
    std::cout << std::setw(50) << "" << get_time() << " reinitialize() mod_b" << std::endl;
    print_status();

    sstr.restore_cout();
  }

  void processing()
  {
    sstr.redirect_cout(this->get_time(), 32);

    std::cout << std::setw(50) << "" << "--------------------------------------------" << std::endl;
    std::cout << std::setw(50) << "" << get_time() << " processing() mod_b" << std::endl;
    print_status();

    sstr.restore_cout();

    out_sc.write(2 * cnt, 0);
    out_sc.write(2 * cnt + 1, 1);

    cnt++;

    for (unsigned long i = 0; i < inp.get_rate(); ++i)
    {
      if (long(inp.read(i)) != in_cnt)
      {
        std::ostringstream str;
        str << " read wrong value for sample: " << i << " value: " << inp.read(i)
            << " expected " << in_cnt << std::endl;
        SC_REPORT_ERROR("TEST", str.str().c_str());
      }
      in_cnt++;
    }

    for (unsigned long i = 0; i < inp.get_rate(); ++i)
    {

      if (is_time_not_equal(out_sc.get_time(i) - last_port_time, out_sc.get_timestep(i)))
      {
        std::ostringstream str;
        str << "Wrong port timestep for out_sc expect: " << out_sc.get_time(i) - last_port_time;
        str << " last_time: " << last_port_time << " current time: " << out_sc.get_time(i);
        str << " get_timestep: " << out_sc.get_timestep(i);
        str << " for sample: " << i;
        SC_REPORT_ERROR("Check", str.str().c_str());
      }

      if (is_time_not_equal(inp.get_time(i) - last_port_time, inp.get_timestep(i)))
      {
        std::ostringstream str;
        str << "Wrong port timestep expect: " << inp.get_time(i) - last_port_time;
        str << " last_time: " << last_port_time << " current time: " << inp.get_time(i);
        str << " get_timestep: " << inp.get_timestep(i);
        str << " for sample: " << i;
        SC_REPORT_ERROR("Check", str.str().c_str());
      }
      last_port_time = inp.get_time(i);

      if (is_time_not_equal(last_port_timestep, out_sc.get_last_timestep(i)))
      {
        std::ostringstream str;
        str << "Wrong port last timestep for out_sc expect: " << last_port_timestep;
        str << " out_sc.get_last_timestep: " << out_sc.get_last_timestep(i);
        str << " for sample: " << i;
        SC_REPORT_ERROR("Check", str.str().c_str());
      }

      if (is_time_not_equal(last_port_timestep, inp.get_last_timestep(i)))
      {
        std::ostringstream str;
        str << "Wrong port last timestep expect: " << last_port_timestep;
        str << " outp.get_last_timestep: " << inp.get_last_timestep(i);
        str << " for sample: " << i;
        SC_REPORT_ERROR("Check", str.str().c_str());
      }
      last_port_timestep = inp.get_timestep(i);
    }

    if (is_time_not_equal(this->get_time() - last_time, this->get_timestep()))
    {
      std::ostringstream str;
      str << "Wrong timestep expect: " << this->get_time() - last_time;
      str << " last_time: " << last_time << " current time: " << this->get_time();
      str << " get_timestep: " << this->get_timestep();
      SC_REPORT_ERROR("Check", str.str().c_str());
    }
    last_time = this->get_time();

    // consistency check
    if (is_time_not_equal(last_time_step, this->get_last_timestep()))
    {
      std::ostringstream str;
      str << "Wrong last timestep expect: " << last_time_step;
      str << " get_last_timestep: " << this->get_last_timestep();
      SC_REPORT_ERROR("Check", str.str().c_str());
    }
    last_time_step = this->get_timestep();
  }

  void change_attributes()
  {
    accept_attribute_changes();

    sstr.redirect_cout(this->get_time(), 42);

    std::cout << std::setw(50) << "" << "--------------------------------------------" << std::endl;
    std::cout << std::setw(50) << "" << get_time() << " change_attributes() mod_b" << std::endl;
    std::cout << std::setw(50) << "" << "     accept_attribute_changes();" << std::endl;
    print_status();

    sstr.restore_cout();
  }

  SCA_CTOR( mod_b ) : inp("inp"), cnt(0), in_cnt(0)
  {
    out_sc.initialize_de_signal(0);
  }

 private:
  long cnt;
  long in_cnt;

  sca_core::sca_time last_time;
  sca_core::sca_time last_time_step;

  sca_core::sca_time last_port_time;
  sca_core::sca_time last_port_timestep;
};

/////////////////////////////////////////////////////////////////////////////

int sc_main( int, char*[] )
{
  TEST_LABEL_START;

  sstr.redirect_cout(sc_core::sc_time_stamp(), 0);

  std::cout << "| mod_a outp|3--->2|inp  mod_b |" << std::endl;

  sstr.restore_cout();

  sca_tdf::sca_signal<double> s1;
  sc_core::sc_signal<long>    s2;

  mod_a a("a");
  a.outp(s1);

  mod_b b("b");
  b.inp(s1);
  b.out_sc(s2);

  sca_util::sca_trace_file* tf1 = sca_util::sca_create_tabular_trace_file("test05_s1.dat");
  sca_util::sca_trace(tf1, s1, "s1");

  sca_util::sca_trace_file* tf2 = sca_util::sca_create_tabular_trace_file("test05_s2.dat");
  sca_util::sca_trace(tf2, s2, "s2");

  sc_core::sc_start();

  sstr.print();

  sca_util::sca_close_tabular_trace_file(tf1);
  sca_util::sca_close_tabular_trace_file(tf2);

  check_trace_file("test05_s1.dat");
  check_trace_file("test05_s2.dat");

  TEST_LABEL_END;

  return 0;
}
