/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

// test ltf_switch from user's guide

#include <systemc-ams>
#include "test_utilities.h"

SCA_TDF_MODULE(ltf_switch)
{
  sca_tdf::sca_in<double> in;
  sca_tdf::sca_out<double> out;

  sca_tdf::sca_de::sca_in<bool> fc_high; // control signal from the discrete-event domain
  
  ltf_switch( sc_core::sc_module_name nm, double fc0_, double fc1_, double h0_ = 1.0)
  : in("in"), out("out"), fc_high("fc_high"), fc0(fc0_), fc1(fc1_), h0(h0_) {}

  void initialize()
  {
    num(0) = 1.0;
    den0(0) = den1(0) = 1.0;
    den0(1) = 1.0 /( 2.0 * M_PI * fc0 );
    den1(1) = 1.0 /( 2.0 * M_PI * fc1 );  }
 
  void processing()
  {
    if ( fc_high.read() )
      out.write( ltf1( num, den1, state, in.read(), h0 ) );
    else
      out.write( ltf0( num, den0, state, in.read(), h0 ) );
    
  }

 private:
  sca_tdf::sca_ltf_nd ltf0, ltf1;
  sca_util::sca_vector<double> num, den0, den1;
  sca_util::sca_vector<double> state;
  double fc0, fc1;
  double h0;
};


SCA_TDF_MODULE(my_step)
{
  sca_tdf::sca_out<double>       out1;
  sca_tdf::sca_de::sca_out<bool> out2;

  SCA_CTOR(my_step) : out1("out1"), out2("out2")
  {}

  void set_attributes()
  {
    out1.set_timestep(1, sc_core::SC_MS);
    out1.set_delay(2);
    out2.set_delay(2);
  }

  void initialize()
  {
    out1.initialize(0.0, 0);
    out1.initialize(0.0, 1);
    out2.initialize(true, 0);
    out2.initialize(true, 1);
  }

  void processing()
  {
    out1.write(1.0);
    out2.write(true);
  }
};

int sc_main( int, char*[] )
{
  TEST_LABEL_START;

  sca_tdf::sca_signal<double> sig1;
  sca_tdf::sca_signal<double> sig2;
  sc_core::sc_signal<bool> sc_sig;

  my_step step("step");
  ltf_switch sw("sw", 1.0e3, 1.0e5, 1.0);

  step.out1(sig1);
  step.out2(sc_sig);
  
  sw.in(sig1);
  sw.out(sig2);  
  sw.fc_high(sc_sig);

  // tracing
  sca_util::sca_trace_file* tf = sca_util::sca_create_tabular_trace_file("ltf_switch.dat");
  sca_util::sca_trace(tf, sig1, "sig1");
  sca_util::sca_trace(tf, sig2, "sig2");
  sca_util::sca_trace(tf, sc_sig, "sc_sig");
  
  // start simulation
  sc_core::sc_start(10, sc_core::SC_MS);

  // stop tracing
  sca_util::sca_close_tabular_trace_file(tf);
  
  test_util::check_results("ltf_switch", 3);

  TEST_LABEL_END;
    
  return 0;
}
  
