/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

// test ltf_nd_filter and ltf_zp_filter from user's guide

#include <systemc-ams>
#include "test_utilities.h"

SCA_TDF_MODULE(ltf_nd_filter)
{
  sca_tdf::sca_in<double> in;
  sca_tdf::sca_out<double> out;

  ltf_nd_filter( sc_core::sc_module_name nm, double fc_, double h0_ = 1.0)
  : in("in"), out("out"), fc(fc_), h0(h0_) {}

  void initialize()
  {
    num(0) = 1.0;
    den(0) = 1.0;
    den(1) = 1.0 / (2.0 * M_PI * fc);
  }
 
  void processing()
  {
    double tmp = ltf_nd( num, den, in.read(), h0 );
    out.write(tmp);
  }

 private:
  sca_tdf::sca_ltf_nd ltf_nd; // Laplace transfer function
  sca_util::sca_vector<double> num, den; // numerator and denominator coefficients
  double fc; // 3dB cutoff frequency in Hz
  double h0; // DC gain
};

SCA_TDF_MODULE(ltf_zp_filter)
{
  sca_tdf::sca_in<double> in;
  sca_tdf::sca_out<double> out;

  ltf_zp_filter( sc_core::sc_module_name nm, double fc_, double h0_ = 1.0)
  : in("in"), out("out"), fc(fc_), h0(h0_) {}

  void initialize()
  {
    // filter requires no zeros to be defined
    poles(0) = sca_util::sca_complex( -2.0 * M_PI * fc, 0.0 );
    k = h0 * 2.0 * M_PI * fc;
  }

  void processing()
  {
    double tmp = ltf_zp( zeros, poles, in.read(), k );
    out.write(tmp);
  }

 private:
  double k; // filter gain
  sca_tdf::sca_ltf_zp ltf_zp; // Laplace transfer function
  sca_util::sca_vector<sca_util::sca_complex> poles, zeros; // poles and zeros as complex values
  double fc; // 3dB cutoff frequency in Hz
  double h0; // DC gain
};

SCA_TDF_MODULE(statespace_eqn)
{
  sca_tdf::sca_in<double> in;
  sca_tdf::sca_out<double> out;

  statespace_eqn( sc_core::sc_module_name nm, double fc_, double h0_ = 1.0)
  : in("in"), out("out"), fc(fc_), h0(h0_) {}

  void initialize()
  {
    double r_val = 1.0e3;
    double c_val = 1.0/(2.0 * M_PI * fc * r_val);

    a(0, 0) = -1.0 / (c_val * r_val);
    b(0, 0) = 1.0 / r_val;
    c(0, 0) = h0 / c_val;
    d(0, 0) = 0.0;
  }

  void processing()
  {
    sca_util::sca_vector<double> x;
    x(0) = in.read();
    
    sca_util::sca_vector<double> y = state_space1( a, b, c, d, s, x );
    out.write(y(0));
  }

 private:
  sca_tdf::sca_ss state_space1;            // state-space equation
  sca_util::sca_matrix<double> a, b, c, d; // state-space matrices
  sca_util::sca_vector<double> s;          // state vector
  double fc; // 3dB cutoff frequency in Hz
  double h0; // DC gain
};


SCA_TDF_MODULE(my_step)
{
  sca_tdf::sca_out<double> out; // output port

  SCA_CTOR(my_step) : out("out")
  {}

  void set_attributes()
  {
    out.set_timestep(1, sc_core::SC_MS);
  }

  void processing()
  {
    out.write(1.0);
  }
};

int sc_main( int, char*[] )
{
  TEST_LABEL_START;

  sca_tdf::sca_signal<double> sig1;
  sca_tdf::sca_signal<double> sig2;
  sca_tdf::sca_signal<double> sig3;
  sca_tdf::sca_signal<double> sig4;

  my_step step("step");
  ltf_nd_filter ltf1("ltf1", 1.0e3, 2.0);
  ltf_zp_filter ltf2("ltf2", 1.0e3, 2.0);
  statespace_eqn stsp("stsp", 1.0e3, 2.0);

  step.out(sig1);

  ltf1.in(sig1);
  ltf1.out(sig2);

  ltf2.in(sig1);
  ltf2.out(sig3);

  stsp.in(sig1);
  stsp.out(sig4);  

  // tracing
  sca_util::sca_trace_file* tf = sca_util::sca_create_tabular_trace_file("ltf_filter");
  sca_util::sca_trace(tf, sig2, "sig2");
  sca_util::sca_trace(tf, sig3, "sig3");
  sca_util::sca_trace(tf, sig4, "sig4");

  sc_core::sc_start(10, sc_core::SC_MS);
  
  sca_util::sca_close_tabular_trace_file(tf);
  
  test_util::check_results("ltf_filter", 3);

  TEST_LABEL_END;
    
  return 0;
}
  
