

/**
 * @file memory_model.hh
 * @author Lasse Lehtonen
 * 
 * @brief Integrates Accurate DRAM model to SCTG2
 *
 * Functions nb_transport_bw(), burstWrite(), burstRead(),
 * handleResponse() and calculateBurstLength() are copied with little
 * modifications from Accurate DRAM Model (LGPL, test_gen.h/.cpp) by
 * Nan Li (KTH).
 *
 */

/*
 * Copyright 2010 Tampere University of Technology
 * 
 *  This file is part of Transaction Generator.
 *
 *  Transaction Generator is free software: you can redistribute it
 *  and/or modify it under the terms of the Lesser GNU General Public
 *  License as published by the Free Software Foundation, either
 *  version 3 of the License, or (at your option) any later version.
 *
 *  Transaction Generator is distributed in the hope that it will be
 *  useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  See the Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public
 *  License along with Transaction Generator.  If not, see
 *  <http://www.gnu.org/licenses/>.
 */

/*
 * $Id: amount.hh 1399 2010-08-26 13:56:45Z lehton87 $
 *
 */

#include "buffer.hh"
#include "configuration.hh"
#include "memory_model.hh"
#include <boost/lexical_cast.hpp>
#include <string>

namespace sctg
{
  
   MemoryModel::MemoryModel(sc_core::sc_module_name name,
			    const boost::property_tree::ptree& pt,
			    sctg::Configuration& config)
      : Resource(name, pt, config),
	req(0),
	resp(0),
	config_(config),
	currentState_(IDLE)
   {    
      
      std::string ocpParam = pt.get<std::string>("<xmlattr>.ocp_param");
      std::string admParam = pt.get<std::string>("<xmlattr>.adm_param");

      id_ = pt.get<unsigned long int>("<xmlattr>.id");
        
      reqSize_ = pt.get<unsigned long int>("<xmlattr>.request_size");

    // Set cycle length    
    cycleLength_ =  sc_core::sc_time(1.0 / getFrequency(), sc_core::SC_US);


    // Create OCP port
    ocpMaster_ = new ocpip::ocp_master_socket_tl1<sizeof(Dt)*8>
      ("master_port", 
       ocpip::ocp_master_socket_tl1<sizeof(Dt)*8>::mm_txn_with_data());

    ocpMaster_->register_nb_transport_bw(this, &MemoryModel::nb_transport_bw);
    ocpMaster_->activate_synchronization_protection();
    
    // Create Accurate DRAM model
    dram_ = new adm_model<MEM_BUSWIDTH>
      ("adm", ocpParam.c_str(), admParam.c_str());

    map_string_type  ocpParamMap;
    readMapFromFile(ocpParam, ocpParamMap);
    ocpip::ocp_parameters params = 
      create_ocp_configuration_from_map("sl", ocpParamMap);
    ocpMaster_->set_ocp_config(params);
    (*ocpMaster_)(dram_->ocpPort);
    
    
    clkgen_ = new sc_core::sc_clock("clkgen", cycleLength_);
    dram_->clk(*clkgen_);
    clk_(*clkgen_);

    SC_METHOD(handleResponse);
    sensitive << clk_.pos();
    dont_initialize();

    SC_THREAD(thread);
    //sensitive << clk_.pos();
    //dont_initialize();


    _buffer->usePackets(true);
  }

   MemoryModel::~MemoryModel()
   {    
   }


   unsigned long int MemoryModel::getReqSize() const
   {
      return reqSize_;
   }


   const sctg::PeMeasurements& MemoryModel::getMeasurements()
   {
      updateMeasurements();

      return measurements_;
   }

   
   void MemoryModel::updateMeasurements()
   {
      measureEnd_   = sc_core::sc_time_stamp();

      switch(currentState_)
      {
	 case IDLE:
	    measurements_.idleTime += 
	       measureEnd_ - measureStart_;	
	    break;
	 case EXECUTING:	
	    measurements_.busyTime += 
	       measureEnd_ - measureStart_;
	    measurements_.execTime += 
	       measureEnd_ - measureStart_;
	    break;
	 case READING:
	    measurements_.busyTime += 
	       measureEnd_ - measureStart_;
	    measurements_.readingTime +=
	       measureEnd_ - measureStart_;
	    break;
	 case CACHE_READ:
	    measurements_.busyTime += 
	       measureEnd_ - measureStart_;
	    measurements_.readingTime +=
	       measureEnd_ - measureStart_;
	    measurements_.cacheTime +=
	       measureEnd_ - measureStart_;
	    break;
	 case WRITING:
	    measurements_.busyTime += 
	       measureEnd_ - measureStart_;
	    measurements_.writingTime +=
	       measureEnd_ - measureStart_;
	    break;
	 case SENDING:
	    measurements_.busyTime += 
	       measureEnd_ - measureStart_;
	    measurements_.sendingTime +=
	       measureEnd_ - measureStart_;
	    break;
	 default:
	    break;
      }

      measurements_.cycleLength = cycleLength_;

      measurements_.idleCycles = 
	 static_cast<unsigned long int>(measurements_.idleTime/cycleLength_);
      measurements_.busyCycles = 
	 static_cast<unsigned long int>(measurements_.busyTime/cycleLength_);
    
      measureStart_ = measureEnd_;

   }



  tlm::tlm_sync_enum MemoryModel::nb_transport_bw(tlm::tlm_generic_payload& tx, 
						  tlm::tlm_phase& phase, 
						  sc_core::sc_time& time)
  {
    if (phase == tlm::BEGIN_RESP) 
      {
	resp = &tx;
	phase = tlm::END_RESP;
	return tlm::TLM_UPDATED;
      } 
    else if ((phase == tlm::END_REQ) || (phase == ocpip::END_DATA)) 
      {
	req = 0;
      } 
    else 
      {
	sc_assert(false);
      }
    
    return tlm::TLM_ACCEPTED;
  }

  
   void MemoryModel::thread()
   {
      sc_core::sc_time startTime;
      sc_core::sc_time endTime;

      while(true)
      {
	 if(!_buffer->rxPacketAvailable())
	 {
	    currentState_ = IDLE;
// 	    std::cout << "MEM " << getName() << " waiting" << std::endl;
	    wait(*(_buffer->rxGetPacketAvailableEvent()));
	    updateMeasurements();
	 }
	 tgPacket* packet = _buffer->rxGetPacket();
	 _buffer->removeToken(packet->size);
	 
	 // Check to which memory area the call belongs to and
	 // calculate the starting address for that memory area
	 unsigned long int area = 0;
	 unsigned long int startAddr = 0;
	 for(unsigned int i = 0; i < memAreas_.size(); ++i)
	 {
	    if((packet->type == CACHE_MISS && memAreas_.at(i)->getId() ==
		packet->dstPort)
	       || 
	       (packet->type != CACHE_MISS && memAreas_.at(i)->getId() == 
		config_.getResourceUserByInPort(packet->dstPort)->getId()))
	    {
// 	       std::cout << "Memory access to area " 
// 			 << memAreas_.at(i)->getName() << std::endl;
	       
	       area = i;
	       break;	       
	    }
	    startAddr += memAreas_.at(i)->getSize();
	 }

	 
	 unsigned long int wordsLeft = 
	    packet->size / (getTerminalWidth(getTerminal())/8);
	 unsigned long int burstWords = 
	    (packet->burstSize) / (getTerminalWidth(getTerminal())/8);
	 burstWords = burstWords < 1 ? 1 : burstWords;
	 unsigned long int maxSize = 
	    burstWords * (getTerminalWidth(getTerminal())/8);
	 
// 	 std::cout << "Words to READ/WRITE " << wordsLeft << std::endl
// 		   << "BurstSize words     " << burstWords << std::endl
// 		   << "Size                " << packet->size << std::endl
// 		   << "RespSize            " << packet->respSize << std::endl
// 		   << "BurstSize           " << packet->burstSize << std::endl;

	 if(packet->type == WRITE_CMD)
	 {
	    currentState_ = WRITING;
	    startTime = sc_core::sc_time_stamp();
	    // Write data in max burst size chunks randomly somewhere between
	    // areas start and end minus burst size in bytes.
	    while(wordsLeft > 0)
	    {
	       unsigned long int addr = startAddr + 
		  (rand()%(memAreas_.at(area)->getSize()-maxSize));
	       unsigned int words = wordsLeft <= burstWords 
		  ? wordsLeft : burstWords;
	       burstWrite(addr, 0, words);
	       wordsLeft -= words;
	    }
	    endTime =  sc_core::sc_time_stamp();
	    
	    if(config_.getMemStream())
	    {
	       **config_.getMemStream()
		  << startTime.value()
		  << ";" << endTime.value()
		  << ";WRITE"
		  << ";" << memAreas_.at(area)->getName()
		  << ";" << memAreas_.at(area)->getId()
		  << ";" << packet->size
		  << ";" << packet->source
		  << ";" << packet->dstPort
		  << std::endl;
	    }
	    updateMeasurements();
	 }
	 else // READ or CACHE_MISS
	 {	    
//  	    std::cout << "READ " 
//  		      << packet->respSize / (getTerminalWidth(getTerminal())/8)
//  		      << " WORDS " << std::endl;

	    // Send responce
	    tgToken token;
	    unsigned long int port;
	    unsigned long int task;
	    unsigned long int size;	 
	    unsigned long int dataLeft = packet->respSize;
	    Resource* pe;

	    token.size = packet->respSize;	    
	    token.type = packet->type;
	    token.source = packet->return_port;
	    token.timeSent = sc_core::sc_time_stamp();
	    
	    startTime = sc_core::sc_time_stamp();

	    if(packet->type == CACHE_MISS)
	    {
	       currentState_ = CACHE_READ;
	       token.destination = 
		  config_.getResource(packet->source)->getTerminalAddress
		  (config_.getResource(packet->source)->getTerminal());
	    }
	    else // READ
	    {
	       currentState_ = READING;
	       token.destination = 
		  config_.getDestination(packet->return_port);
	       task = config_.getResourceUserByInPort(token.destination)->
		  getId();
	    }	       


	    if(respMap.find(packet->id) == respMap.end())
	    {	       	       
	       Resp r = {token.size, packet->packets, config_.getTokenId()};
	       respMap[packet->id] = r;
	       token.id = respMap[packet->id].tokenId;

	       if(packet->type == CACHE_MISS)
	       {
		  config_.getBufferByResource
		     (config_.getResource(packet->source)->getId())
		     ->expectToken(token);
	       }
	       else // READ
	       {
		  config_.getBufferByResource
		     (config_.getResourceByResourceUser(task)->getId())
		     ->expectToken(token);	    
		  config_.addSendTime(token.source);
	       }	       
	       	       
	    }

	    token.id = respMap[packet->id].tokenId;

	    if(respMap[packet->id].respLeft > 1)
	    {
	       dataLeft = packet->respSize / packet->packets;
	       respMap[packet->id].dataLeft -= dataLeft;
	       respMap[packet->id].respLeft--;
	    }
	    else
	    {
	       dataLeft = respMap[packet->id].dataLeft;
	       respMap.erase(respMap.find(packet->id));
	    }
	    	    
// 	    std::cout << "MEM " << getName() << " sending token with size "
// 		      << dataLeft << std::endl;

	    while(wordsLeft > 0)
	    {
	       unsigned int words = wordsLeft <= burstWords 
		  ? wordsLeft : burstWords;
	       burstRead(startAddr + 
			 (rand()%(memAreas_.at(area)->getSize()-maxSize)), 
			 words, true);
	       wordsLeft -= words;	
// 	       std::cout << " READING " << words << " words from mem" 
// 			 << std::endl;
	    }

	    wait(readDone);

	    while(dataLeft > 0)
	    {
	       

	       // Create packet
	       tgPacket* pkt = new tgPacket;
	       if(_packetSize > 0) // Zero packet_size means "don't cut tokens"
	       {
		  size = dataLeft < _packetSize ? dataLeft : _packetSize;
	       }
	       else
	       {
		  size = dataLeft;
	       }
	       dataLeft -= size;
// 	       std::cout << "dataLeft " << dataLeft << std::endl;
	       size = size < 4 ? 4 : size; // Four bytes is minimum packet size
	       size = (size % 4 == 0) ? size : size + (4-(size % 4));
	       pkt->size = size;
	       
	       if(packet->type == CACHE_MISS)
	       {
		  pkt->address = token.destination;
	       }
	       else // READ
	       {
		  port = token.destination;
		  task = config_.getResourceUserByInPort(port)->getId();
		  pe = config_.getResourceByResourceUser(task);
		  pkt->address = pe->getTerminalAddress(pe->getTerminal());
	       }
	       pkt->id   = token.id;

//  	       std::cout << "MEM response to port " << port << " task "
//  			 << task << " in address " << std::hex
//  			 << pkt->address << std::dec 
//  			 << " with size " << size << std::endl;
	       
	       
	       updateMeasurements();
	       while(_buffer->txSpaceLeft() < pkt->size)
	       {			
		  currentState_ = SENDING;
		  wait(*(_buffer->txGetReadEvent()));
		  updateMeasurements();
	       }
	    
	       _buffer->txPutPacket(pkt);
	    }

	    endTime =  sc_core::sc_time_stamp();
	    
	    if(config_.getMemStream())
	    {
	       if(packet->type == CACHE_MISS)
	       {
		  **config_.getMemStream()
		     << startTime.value()
		     << ";" << endTime.value()
		     << ";CACHE MISS"
		     << ";" << memAreas_.at(area)->getName()
		     << ";" << memAreas_.at(area)->getId()
		     << ";" << token.size
		     << ";" << token.source
		     << ";" << token.destination
		     << std::endl;
	       }
	       else
	       {
		  **config_.getMemStream()
		     << startTime.value()
		     << ";" << endTime.value()
		     << ";READ"
		     << ";" << memAreas_.at(area)->getName()
		     << ";" << memAreas_.at(area)->getId()
		     << ";" << token.size
		     << ";" << token.source
		     << ";" << token.destination
		     << std::endl;
	       }
	    }

//  	    std::cout << " DONE READING" << std::endl;
	 }
	 
	 delete packet->data;
	 delete packet;
      }

   }
   
  void MemoryModel::burstWrite(uint64 address, unsigned int threadId, 
			       unsigned int length, bool posted, 
			       bool srmd)
  {
    tlm::tlm_generic_payload *txn = ocpMaster_->get_transaction();
    ocpMaster_->reserve_data_size(*txn, sizeof(Dt) * length);
    txn->set_address(address);
    txn->set_byte_enable_ptr(NULL);
    txn->set_streaming_width(sizeof(Dt));
    txn->set_response_status(tlm::TLM_INCOMPLETE_RESPONSE);
    txn->set_command(tlm::TLM_WRITE_COMMAND);

    /* set extension for posted */
    if (posted) ocpMaster_->validate_extension<ocpip::posted>(*txn);

    /* set extension for srmd */
    if (srmd) ocpMaster_->validate_extension<ocpip::srmd>(*txn);

    /* set extension burst_length */
    ocpip::burst_length *bLen;
    ocpMaster_->get_extension(bLen, *txn);
    bLen->value = length;
    ocpMaster_->validate_extension<ocpip::burst_length>(*txn);

    /* set extension thread_id */
    ocpip::thread_id *tId;
    ocpMaster_->get_extension(tId, *txn);
    tId->value = threadId;
    ocpMaster_->validate_extension<ocpip::thread_id>(*txn);

    if (srmd) {
      tlm::tlm_phase phase = tlm::BEGIN_REQ;
      sc_time time = SC_ZERO_TIME;
      req = txn;

      cout << "[REQ @ " << sc_time_stamp() << "] [WRITE] BURST(" << length << ") - SRMD : request sent." << endl;
      (*ocpMaster_)->nb_transport_fw(*req, phase, time);
      
      while (req) wait(clkgen_->posedge_event());

    }


    /* send data */
    for (unsigned i = 0; i < length; ++i) {
      *((Dt *)txn->get_data_ptr() + i) = 1234;	/* data does not matter */
      
      tlm::tlm_phase phase;
      if (srmd)
	phase = ocpip::BEGIN_DATA;
      else
	phase = tlm::BEGIN_REQ;
      sc_time time = SC_ZERO_TIME;

      req = txn;

      if ((*ocpMaster_)->nb_transport_fw(*req, phase, time) == tlm::TLM_UPDATED)
	{
	  req = 0;
	  wait(clkgen_->posedge_event());
	} 
      else 
	{
	  while (req) wait(clkgen_->posedge_event());
	}
      
    }

    if (posted)
      ocpMaster_->release_transaction(txn);
  }

  void MemoryModel::burstRead(uint64 address, unsigned int threadId, 
			      unsigned int length, bool srmd)
  {
    tlm::tlm_generic_payload *txn = ocpMaster_->get_transaction();
    ocpMaster_->reserve_data_size(*txn, length * sizeof(Dt));
    txn->set_address(address);
    txn->set_byte_enable_ptr(0);
    txn->set_streaming_width(sizeof(Dt));
    txn->set_response_status(tlm::TLM_INCOMPLETE_RESPONSE);
    txn->set_command(tlm::TLM_READ_COMMAND);

    /* set extension for srmd */
    if (srmd) ocpMaster_->validate_extension<ocpip::srmd>(*txn);

    /* set extension burst_length */
    ocpip::burst_length *bLen;
    ocpMaster_->get_extension(bLen, *txn);
    bLen->value = length;
    ocpMaster_->validate_extension<ocpip::burst_length>(*txn);

    /* set extension thread_id */
    ocpip::thread_id *tId;
    ocpMaster_->get_extension(tId, *txn);
    tId->value = threadId;
    ocpMaster_->validate_extension<ocpip::thread_id>(*txn);

    if (srmd) {
      tlm::tlm_phase phase = tlm::BEGIN_REQ;
      sc_time time = SC_ZERO_TIME;
      req = txn;


      (*ocpMaster_)->nb_transport_fw(*req, phase, time);

      while (req) wait(clkgen_->posedge_event());
      
    } 
    else 
      {
	for (unsigned i = 0; i < length; ++i) {
	  tlm::tlm_phase phase = tlm::BEGIN_REQ;
	  sc_time time = SC_ZERO_TIME;
	  req = txn;

	  (*ocpMaster_)->nb_transport_fw(*req, phase, time);

	  while (req) wait(clkgen_->posedge_event());

      }
    }
  }

  void MemoryModel::handleResponse()
  {
    if (!resp) return;
    static std::map<tlm::tlm_generic_payload *, unsigned int> 
      burstLengthMap, burstCountMap;

    if (!burstLengthMap[resp]) {
      /* first response */
      burstLengthMap[resp] = calculateBurstLength(*resp);
      burstCountMap[resp] = 0;
    }


    if (resp->get_command() == tlm::TLM_WRITE_COMMAND) {
      if (ocpMaster_->get_extension<ocpip::srmd>(*resp)) {
	cout << "[RESP @ " << sc_time_stamp() << "] [WRITE] BURST(" 
	     << burstLengthMap[resp] << ") - SRMD : response received." << endl;
	ocpMaster_->release_transaction(resp);
	burstLengthMap.erase(resp);
	burstCountMap.erase(resp);

	cout << "[TXN " << "] [WRITE] [END @ " << "]" << endl;
	txnIdMap.erase(resp);
      } else {
// 	cout << "[RESP @ " << sc_time_stamp() << "] [WRITE] BURST(" 
// 	     << burstLengthMap[resp] << ") : DATA[" << burstCountMap[resp] 
// 	     << "] response received." << endl;
	if (burstCountMap[resp] == burstLengthMap[resp] - 1) {
// 	  cout << "[FIN @ " << sc_time_stamp() << "] [WRITE] BURST(" 
// 	       << burstLengthMap[resp] << ") : finished." << endl;
	  burstLengthMap.erase(resp);
	  burstCountMap.erase(resp);
	  ocpMaster_->release_transaction(resp);
	}
      }

    } else {
      // Td data = *((Td *)resp->get_data_ptr() + burstCountMap[resp]);
//       cout << "[RESP @ " << sc_time_stamp() << "] [READ] BURST(" << burstLengthMap[resp] << ") : DATA[" << burstCountMap[resp] << "] = " << endl;

      if (burstCountMap[resp] == burstLengthMap[resp] - 1) {
// 	cout << "[FIN @ " << sc_time_stamp() << "] [READ] BURST(" << burstLengthMap[resp] << ") : finished." << endl;
	burstLengthMap.erase(resp);
	burstCountMap.erase(resp);
	ocpMaster_->release_transaction(resp);
	
	txnIdMap.erase(resp);
	readDone.notify(sc_core::SC_ZERO_TIME);
      }
    }

    if (burstCountMap.count(resp))
      burstCountMap[resp]++;
    resp = 0;
  }

  unsigned int MemoryModel::calculateBurstLength(tlm::tlm_generic_payload &txn)
  {
    ocpip::burst_length *bLen;
    if (ocpip::extension_api::get_extension<ocpip::burst_length>(bLen, txn))
      return bLen->value;
    else {
      SC_REPORT_WARNING(SC_ID_WITHOUT_MESSAGE_, 
			"burst_length extension is not used. "
			"Calculating burst length from data length.");
      return txn.get_data_length() / sizeof(Dt);
    }
  }

   void MemoryModel::mapResourceUser(ResourceUser* task)
   {
      std::cout << "MemArea " << task->getName() << " mapped to Memory "
		<< getName() << std::endl;
      memAreas_.push_back(dynamic_cast<MemArea*>(task));
   }
  
}

// Local Variables:
// mode: c++
// c-file-style: "ellemtel"
// c-basic-offset: 3
// End:


