/**
 *
 * @file processing_element.hh
 * @author Lasse Lehtonen
 *
 *
 */

/*
 * Copyright 2010 Tampere University of Technology
 * 
 *  This file is part of Transaction Generator.
 *
 *  Transaction Generator is free software: you can redistribute it
 *  and/or modify it under the terms of the Lesser GNU General Public
 *  License as published by the Free Software Foundation, either
 *  version 3 of the License, or (at your option) any later version.
 *
 *  Transaction Generator is distributed in the hope that it will be
 *  useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  See the Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public
 *  License along with Transaction Generator.  If not, see
 *  <http://www.gnu.org/licenses/>.
 */

/*
 * $Id: processing_element.hh 1916 2011-07-06 12:44:26Z lehton87 $
 *
 */


#ifndef SCTG_PROCESSING_ELEMENT_HH
#define SCTG_PROCESSING_ELEMENT_HH

#include "resource.hh"
#include "amount.hh"
#include "task.hh"
#include "buffer.hh"
#include "common.hh"

#include <systemc>
#include <boost/property_tree/ptree.hpp>
#include <memory>
#include <map>

namespace sctg
{
   /** Models a processor or hardare accelerator
    *
    */
   class ProcessingElement : public Resource
   {
   public:

      
      SC_HAS_PROCESS(ProcessingElement);

      /** Constructor
       */
      ProcessingElement(sc_core::sc_module_name name, 
			const boost::property_tree::ptree& pt,
			const boost::property_tree::ptree& peLib,
			sctg::Configuration& config);
    
      /** Destructor
       */
      virtual ~ProcessingElement();

      /** Main execution thread which consumes simulation time
       */
      void thread();

      /** How many integer operations are excecuted in clock cycle
       */
      double getIntOps();
      /** How many floating point operations are excecuted in clock cycle
       */
      double getFloatOps();
      /** How many memory operations are excecuted in clock cycle
       */
      double getMemOps();

      /** Return receive cost (clock cycles) for intra group communication
       */
      unsigned long int getIntraGroupRxCost();
      /** Return send cost (clock cycles) for intra group communication
       */
      unsigned long int getIntraGroupTxCost();
      /** Return receive cost (clock cycles) for inter group communication
       */
      unsigned long int getInterGroupRxCost();
      /** Return send cost (clock cycles) for inter group communication
       */
      unsigned long int getInterGroupTxCost();
      /** Return receive cost (clock cycles) for inter resource communication
       */
      unsigned long int getInterPeRxCost();
      /** Return send cost (clock cycles) for inter resource communication
       */
      unsigned long int getInterPeTxCost();
    
      /** True if PE has task with this in_port
       */
      bool hasInPort(unsigned long int port);

      /** PE receives a token from event
       */
      void receiveEvent(tgToken token);

      /** Returns measurement information
       */
      const sctg::PeMeasurements& getMeasurements();

      /** Return tokens 
       */
      std::queue<tgToken>& getTokenQueue(); 
      
      /** Returns average utilization from the beginning to this point
       */
      double getAvgUtilization();

      /** Updates latency for tokens that are not completely sent
       */
      void updateUnfinishedTokensLatency();

      /** Assigns Task to this PE
       */
      void mapResourceUser(ResourceUser* task);

   private:

      /** Returns next task to be executed
       */
      sctg::Task* schedule();

      /** updates measurements
       */
      void updateMeasurements();

      /** TX DMA handler
       */
      void txDma();

      /** RX DMA handler
       */
      void rxDma();

      // Performance factors (operations per clock cycle)
      double _intOps;
      double _floatOps;
      double _memOps;    

      // Communication costs
      Amount<unsigned long int>* _intraGroupRxCost;
      Amount<unsigned long int>* _intraGroupTxCost;
      Amount<unsigned long int>* _interGroupRxCost;
      Amount<unsigned long int>* _interGroupTxCost;
      Amount<unsigned long int>* _interPeRxCost;
      Amount<unsigned long int>* _interPeTxCost;

      // Whether DMA is enabled or not
      bool _dmaEnabled;

      // List of tasks on this PE      
      std::queue<Task*> _fifoScheduler;

      Configuration& _config;
      
      std::map<unsigned long int, tgToken> _sendTokens;

      sc_core::sc_time  _cycleLength;

      PeMeasurements _measurements;

      sc_core::sc_event _eventEvent;

      Task* _currentTask;

      enum PeState {IDLE, EXECUTING, SENDING, READING, INTRA_TX_WAIT,
		    TX_WAIT, RX_WAIT, I_CACHE_MISSING, D_CACHE_MISSING};
      PeState _currentState;
    
      sc_core::sc_time _measureStart;
      sc_core::sc_time _measureEnd;

      std::queue<tgPacket*> _txDmaQueue;
      sc_core::sc_event     _txDmaEvent;

      std::queue<tgPacket*> _rxDmaQueue;
      sc_core::sc_event     _rxDmaEvent;

      std::queue<tgToken>   _tokenQueue;

      std::vector<Task*>    _tasks;

      Amount<double>* _iCacheMissInterval;
      Amount<double>* _dCacheMissInterval; 
      unsigned long int _dLineSize;
      unsigned long int _iLineSize;
      unsigned long int _iMemId;
      unsigned long int _dMemId;
      unsigned long int _nextIMiss;
      unsigned long int _nextDMiss;

   };
}

#endif


// Local Variables:
// mode: c++
// c-file-style: "ellemtel"
// c-basic-offset: 3
// End:
