/**
 *
 * @file fifo.hh
 * @author Lasse Lehtonen
 *
 *
 */

/*
 * Copyright 2010 Tampere University of Technology
 * 
 *  This file is part of Transaction Generator.
 *
 *  Transaction Generator is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Transaction Generator is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Transaction Generator.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * $Id: fifo.hh 1399 2010-08-26 13:56:45Z lehton87 $
 *
 */

#ifndef ASEBT_SC_RTL_2_FIFO_HH
#define ASEBT_SC_RTL_2_FIFO_HH

#include <systemc>
#include <cmath>

namespace asebt
{
namespace sc_rtl_2
{

   template< int data_width_g, int depth_g>
   class fifo : public sc_core::sc_module
   {
   public:
      sc_core::sc_in_clk clk;
      sc_core::sc_in<bool> rst_n;

      sc_core::sc_in<sc_dt::sc_bv<data_width_g> > data_in;
      sc_core::sc_in<bool> we_in;
      sc_core::sc_out<bool> full_out;
      sc_core::sc_out<bool> one_p_out;
  
      sc_core::sc_in<bool> re_in;
      sc_core::sc_out<sc_dt::sc_bv<data_width_g> > data_out;
      sc_core::sc_out<bool> empty_out;
      sc_core::sc_out<bool> one_d_out;

      SC_HAS_PROCESS(fifo);

      fifo(sc_core::sc_module_name name)
	 : sc_core::sc_module(name),
	   clk("clk"),
	   rst_n("rst_n"),
	   data_in("data_in"),
	   we_in("we_in"),
	   full_out("full_out"),
	   one_p_out("one_p_out"),
	   re_in("re_in"),
	   data_out("data_out"),
	   empty_out("empty_out"),
	   one_d_out("one_d_out"),
	   full_r("full_r"),
	   empty_r("empty_r"),
	   one_d_r("one_d_r"),
	   one_p_r("one_p_r"),
	   data_amount_r("data_amount_r"),
	   in_ptr_r("in_ptr_r"),
	   out_ptr_r("out_ptr_r")
      {
	 for(unsigned int i = 0; i < depth_g; ++i)
	 {
	    fifo_buffer_r[i].write("0");
	 }

	 SC_METHOD(method);
	 sensitive << clk.pos() << rst_n;

	 SC_METHOD(output);
	 sensitive << out_ptr_r << full_r
		   << empty_r << one_d_r
		   << one_p_r;
	 for(unsigned int i = 0; i < depth_g; ++i)
	 {
	    sensitive << fifo_buffer_r[i];
	 }

	 empty_out.initialize(true);

      }
  
      virtual ~fifo()
      {
      }

      void output()
      {
	 data_out.write(fifo_buffer_r[out_ptr_r.read()].read());
	 full_out.write(full_r);
	 empty_out.write(empty_r);
	 one_d_out.write(one_d_r);
	 one_p_out.write(one_p_r);
      }

      void method()
      {
	 if(rst_n.read() == 0)
	 {
	    full_r.write(false);
	    empty_r = true;
	    one_d_r = false;
	    in_ptr_r = 0;
	    out_ptr_r = 0;
	    data_amount_r = 0;
	    one_p_r = depth_g == 1 ? true : false;
	 }
	 else
	 {	
	    if(we_in.read() == true && 
	       re_in.read() == false)
	    {
	       if(full_r.read() == false)
	       {
		  empty_r = false;
		  if(in_ptr_r.read() == depth_g - 1) 
		  { in_ptr_r = 0; }
		  else 
		  { in_ptr_r = in_ptr_r.read() + 1; }
		  data_amount_r = data_amount_r.read() + 1;
		  fifo_buffer_r[in_ptr_r.read()] = data_in;
	       }

	       if(data_amount_r.read() + 2 == depth_g)
	       {
		  full_r = false;
		  one_p_r = true;
	       }
	       else if(data_amount_r.read() + 1 == depth_g)
	       {
		  full_r = true;
		  one_p_r = false;
	       }
	       else
	       {
		  full_r = false;
		  one_p_r = false;
	       }

	       if(empty_r.read() == 1)
	       { one_d_r = true; }
	       else
	       { one_d_r = false; }	  

	    }
      
	    else if(we_in.read() == false && re_in.read() == 1)
	    {
	       if(empty_r.read() == false)
	       {
		  if(out_ptr_r.read() == depth_g - 1)
		  {
		     out_ptr_r = 0;
		  }
		  else
		  {
		     out_ptr_r = out_ptr_r.read() + 1;
		  }
	      
		  full_r = false;
		  data_amount_r = data_amount_r.read() - 1;
	      
		  if(data_amount_r.read() == 2)
		  {
		     empty_r = false;
		     one_d_r = true;
		  }
		  else if(data_amount_r.read() == 1)
		  {
		     empty_r = true;
		     one_d_r = false;
		  }
		  else
		  {
		     empty_r = false;
		     one_d_r = false;
		  }
	      
		  if(full_r.read() == true)
		  { one_p_r = true; }
		  else
		  { one_p_r = false; }	    
	      
	       }
	    }
	    else if(we_in.read() == true && 
		    re_in.read() == true)
	    {
	       if(full_r.read() == false && 
		  empty_r == false)
	       {
		  if(in_ptr_r.read() == depth_g - 1)
		  { in_ptr_r = 0; }
		  else 
		  { in_ptr_r = in_ptr_r.read() + 1;}
	      
		  if(out_ptr_r.read() == depth_g - 1)
		  { out_ptr_r = 0; }
		  else 
		  { out_ptr_r = out_ptr_r.read() + 1;}
	      
		  full_r = false;
		  empty_r = false;
	      
		  fifo_buffer_r[in_ptr_r.read()] = data_in;
	       }
	       else if(full_r.read() == true && 
		       empty_r.read() == false)
	       {
		  if(out_ptr_r.read() == depth_g - 1)
		  { out_ptr_r = 0; }
		  else 
		  { out_ptr_r = out_ptr_r.read() + 1;}
	      
		  full_r = false;
		  one_p_r = true;
		  data_amount_r = data_amount_r.read() - 1;
	      
		  if(data_amount_r.read() == 2)
		  {
		     empty_r = false;
		     one_d_r = true;
		  }
		  else if(data_amount_r.read() == 1)
		  {
		     empty_r = true;
		     one_d_r = false;
		  }
		  else
		  {
		     empty_r = false;
		     one_d_r = false;
		  }
	       }
	       else if(full_r.read() == false && 
		       empty_r.read() == true)
	       {
		  if(in_ptr_r.read() == depth_g - 1)
		  { in_ptr_r = 0; }
		  else 
		  { in_ptr_r = in_ptr_r.read() + 1;}
	      
		  empty_r = false;
		  one_d_r = true;
		  fifo_buffer_r[in_ptr_r.read()] = data_in;
		  data_amount_r = data_amount_r.read() + 1;
	      
		  if(data_amount_r.read() == 2)
		  {
		     empty_r = false;
		     one_d_r = true;
		  }
		  else if(data_amount_r.read() == 1)
		  {
		     empty_r = true;
		     one_d_r = false;
		  }
		  else
		  {
		     empty_r = false;
		     one_d_r = false;
		  }
	       }
	    }
	 }    
      }


   private:

      sc_core::sc_signal<bool> full_r;
      sc_core::sc_signal<bool> empty_r;
      sc_core::sc_signal<bool> one_d_r;
      sc_core::sc_signal<bool> one_p_r;
      sc_core::sc_signal<sc_dt::sc_uint<16> > data_amount_r;
      sc_core::sc_signal<sc_dt::sc_uint<16> > in_ptr_r;
      sc_core::sc_signal<sc_dt::sc_uint<16> > out_ptr_r;

      sc_core::sc_signal<sc_dt::sc_bv<data_width_g> > fifo_buffer_r[depth_g];

   };

}
}

#endif


// Local Variables:
// mode: c++
// c-file-style: "ellemtel"
// c-basic-offset: 3
// End:
