// Copyright (c) 2004 Sonics, Inc.
//
// Confidential and Proprietary Information of Sonics, Inc.
// Use, disclosure, or reproduction is prohibited without
// written permission from Sonics, Inc.
//
// $Id: StlTransfer.cc,v 1.1 2007/01/25 22:09:07 halexan Exp $

// Implementation of the templated StlTransferCommand class. This file
// is separated from other non templated command classes declared in
// StlCommand.h and implemented in StlCommand.cc
#include "StlCommand.h"
#include <sstream>
#include <cmath>

using namespace std;
using namespace OcpIp;

#define MEMDATABUF_RESERVE_INCR 256
//////////////////////////////////////////////////////////////////
// FUNCTION: StlParserTransferCommand::StlParserTransferCommand
// DESCRIPTION: constructor. Sets starting values and allocates
// data memory
// ARGUMENTS: None
// RETURNS: Nothing
//////////////////////////////////////////////////////////////////
template<typename Td, typename Ta>
StlParserTransferCommand<Td,Ta>::StlParserTransferCommand(
    const OcpParams& params ) :
    StlTransferCommand<Td,Ta>(),
    m_currentData( 0 ),
    m_currentDataMask( 0 ),
    m_currentDataInfo( 0 ),
    m_ocpParams( params ),
    m_pDataInfo( NULL ),
    m_pByteEn( NULL ),
    m_numDataReserved( 0 ),
    m_defaultThreadId( 0 ),
    m_dataIndex( -1 )
{
    reserve( MEMDATABUF_RESERVE_INCR );
    assert( m_numDataReserved == MEMDATABUF_RESERVE_INCR  );
}

//////////////////////////////////////////////////////////////////
// FUNCTION: StlParserTransferCommand::reserve
// DESCRIPTION: Function top increase the size of the allocated
// memory. The storage has to stay contiguous. This is a fairly
// expensive operation, use in moderation.
// ARGUMENTS: absolute size of the reserve desired
// RETURNS: Nothing
//////////////////////////////////////////////////////////////////
template<typename Td, typename Ta>
void
StlParserTransferCommand<Td,Ta>::reserve( uint32_t numData )
{
    if ( numData == m_numDataReserved ) return;
    assert( numData > m_numDataReserved );
    
    DataType*     pNewData     = new DataType[numData];
    DataInfoType* pNewDataInfo = new DataInfoType[numData];
    ByteEnType*   pNewByteEn   = new ByteEnType[numData];
    _dataMask().resize( numData, 0 );
    _dataInfoMask().resize( numData, 0 );

    // Initialize to 0 and copy the old buffers to the new one
    for ( uint32_t i=0; i <numData; ++i ) pNewData    [i] = 0;
    for ( uint32_t i=0; i <numData; ++i ) pNewDataInfo[i] = 0;
    for ( uint32_t i=0; i <numData; ++i ) pNewByteEn  [i] = 0;

    if ( this->request.MDataPtr != NULL )
        for ( uint32_t i=0; i < m_numDataReserved; ++i )
            pNewData    [i] = this->request.MDataPtr[i];
    if ( m_pDataInfo != NULL )
        for ( uint32_t i=0; i < m_numDataReserved; ++i )
            pNewDataInfo[i] = m_pDataInfo[i];
    if ( m_pByteEn != NULL )
        for ( uint32_t i=0; i < m_numDataReserved; ++i )
            pNewByteEn[i] = m_pByteEn[i];

    DELETE_ARRAY(_request().MDataPtr);
    DELETE_ARRAY(m_pDataInfo);
    DELETE_ARRAY(m_pByteEn);
    _request().MDataPtr = pNewData;
    m_pDataInfo         = pNewDataInfo;
    m_pByteEn           = pNewByteEn;
    m_numDataReserved   = numData;
}

//////////////////////////////////////////////////////////////////
// FUNCTION: StlParserTransferCommand::clear
// DESCRIPTION: Interface to the parser. Used to reset the data
// ARGUMENTS: None
// RETURNS: Nothing
//////////////////////////////////////////////////////////////////
template<typename Td, typename Ta>
void
StlParserTransferCommand<Td,Ta>::clear()
{
    Request& request = _request();
    // do not use request.reset() it NULLs the data pointer!
    request.MCmd = OCP_MCMD_IDLE;
    request.MAddr = 0;
    request.DataLength = 1;
    request.MAddrSpace = 0;
    request.MByteEn = 0xFFFFFFFF;
    request.MReqInfo = 0;
    request.MDataInfo = 0;
    request.MThreadID = m_defaultThreadId;
    request.MConnID = 0;
    request.MBurstLength = 1;
    request.MBurstPrecise = true;
    request.MBurstSeq = OCP_MBURSTSEQ_INCR;
    request.MBurstSingleReq = false;

    // ByteEn, DataInfo pointers
    request.MByteEnPtr    = NULL; // (by default use MByteEn)
    request.MDataInfoPtr  = NULL; // (by default use MDataInfo)
    _idleCycles()         = 0;
    m_dataIndex           = -1;
    m_currentData         = 0;
    m_currentDataMask     = 0;
    m_currentDataInfo     = static_cast<DataInfoType>( -1 );
    m_currentDataInfoMask = 0;
    m_currentDataSpec     = NODATASPEC;
    m_currentDataInfoSpec = NODATAINFOSPEC;
    m_randByteEn          = false;
    m_transferWidth       = ocpParam( P_data_wdth );
}

//////////////////////////////////////////////////////////////////
// FUNCTION: StlParserTransferCommand::addData
// DESCRIPTION: Interface to the parser. Used to manage the data
// memory
// ARGUMENTS: data to add. Overloaded with versions for byte enable
// and data info as well. The memory for these 3 is managed centrally,
// so these functions are the only interface to add data/byteen/datainfo
// RETURNS: Nothing
//////////////////////////////////////////////////////////////////
template<typename Td, typename Ta>
void
StlParserTransferCommand<Td,Ta>::addData( const DataType& data,
                                          const DataInfoType& dataInfo)
{
    ++m_dataIndex;

    if ( m_dataIndex >= static_cast<int>( m_numDataReserved ))
        reserve( m_numDataReserved + MEMDATABUF_RESERVE_INCR );
    
    assert( m_dataIndex >= 0 );
    assert( m_dataIndex < static_cast<int>( m_numDataReserved ));
    _request().MDataPtr[m_dataIndex] = data;
    m_pDataInfo[m_dataIndex] = dataInfo;
    _request().MDataInfoPtr = m_pDataInfo;
}

template<typename Td, typename Ta>
void
StlParserTransferCommand<Td,Ta>::addData( const DataType& data,
                                          const DataInfoType& dataInfo,
                                          const ByteEnType& byteen)
{
    addData( data, dataInfo );
    m_pByteEn[m_dataIndex] = byteen;
    _request().MByteEnPtr = m_pByteEn;
}
    
