#include "simpleMaster.h"

#define NUM_TESTS         9

// -------------------------------------------------------------------
// constructor
// -------------------------------------------------------------------
template<typename TdataCl>
Master<TdataCl>::Master(
    sc_module_name name,
    int            id,
    ostream*       debug_os_ptr
) : sc_module(name),
    ipP("ipPort"),
    clk("clkPort"),
    m_ID(id),
    m_debug_os_ptr(debug_os_ptr),
    m_threads(1),
    m_addrspace(false),
    m_sthreadbusy(false),
    m_sthreadbusy_exact(false),
    m_mthreadbusy(false),
    m_mthreadbusy_exact(false),
    m_respaccept(true),
    m_datahandshake(false),
    m_writeresp_enable(false),
    m_writenonpost_enable(false),
    m_respaccept_fixeddelay(1),
    m_respaccept_delay(1)
{
    // setup a SystemC thread process, which uses dynamic sensitive
    SC_THREAD(requestThreadProcess);
    sensitive<<clk.pos();

    // setup a SystemC thread process, which uses dynamic sensitive
    SC_THREAD(responseThreadProcess);
    sensitive<<clk.pos();

    // setup a SystemC thread process to drive any connected sideband signals
    SC_THREAD(exerciseSidebandThreadProcess);
    sensitive<<clk.pos();
    
    //assuming default timing the master will expect sthreadbusy to be stable 1 ps after the clock edge    
    m_sthreadbusy_sample_time=sc_time(1, SC_PS);
}

template<typename TdataCl>
void Master<TdataCl>::provideChannelConfiguration(MapStringType& passedMap){
  m_parameters.setOCPConfiguration(name(), passedMap);
}

template<typename TdataCl>
void Master<TdataCl>::setModuleConfiguration(MapStringType& passedMap){
    if (!(OCPParameters::getBoolOCPConfigValue("", "mrespaccept_fixeddelay", 
            m_respaccept_fixeddelay, passedMap)) ) {
        // Could not find the parameter so we must set it to a default
#ifdef DEBUG 
        cout << "Warning: paramter \"" << "mrespaccept_fixeddelay" 
               << "\" was not found in the module parameter map." << endl;
        cout << "         setting missing parameter to "<<m_respaccept_fixeddelay<<"." << endl;
#endif
    }

    if (!(OCPParameters::getIntOCPConfigValue("", "mrespaccept_delay", 
            m_respaccept_delay, passedMap)) ) {
        // Could not find the parameter so we must set it to a default
#ifdef DEBUG 
        cout << "Warning: paramter \"" << "mrespaccept_delay" 
               << "\" was not found in the module parameter map." << endl;
        cout << "         setting missing parameter to "<<m_respaccept_delay<<"." << endl;
#endif
    }
}

// -------------------------------------------------------------------
// SystemC Method Master::end_of_elaboration()
// -------------------------------------------------------------------
// 
//  At this point, everything has been built and connected.
//  We are now free to get our OCP parameters and to set up our
//  own variables that depend on them.
//
template<typename TdataCl>
void Master<TdataCl>::end_of_elaboration()
{
    // Call the System C version of this function first
    sc_module::end_of_elaboration();

    //-----------------------------------------
    //  OCP Parameters
    //-----------------------------------------

    // Get the number of threads
    m_threads=m_parameters.threads;

    // This Reference Master is single threaded.
    if (m_threads > 1) {
        cout << "ERROR: Single threaded Master \"" << name() 
                 << "\" connected to OCP with " << m_threads 
                 << " threads." << endl; 
    }

    // is the MAddrSpace field part of the OCP channel?
    m_addrspace=m_parameters.addrspace;

    // is SThreadBusy part of the channel?
    m_sthreadbusy=m_parameters.sthreadbusy;

    // Is SThreadBusy compliance required?
    m_sthreadbusy_exact=m_parameters.sthreadbusy_exact;

    // is MThreadBusy part of the channel?
    m_mthreadbusy=m_parameters.mthreadbusy;

    // Is MThreadBusy compliance required?
    m_mthreadbusy_exact=m_parameters.mthreadbusy_exact;

    // is MRespAccept part of the channel?
    m_respaccept=m_parameters.respaccept;

    // Just a double check here
    if (m_mthreadbusy_exact && m_respaccept) {
        cout << "ERROR: Master \"" << name() 
        << "\" connected to OCP with both MThreadBusy_Exact and MRespAccept active which are exclusive." << endl; 
    }

    // is Data Handshake part of the channel?
    m_datahandshake=m_parameters.datahandshake;
    // if so, quit. This core does not support it.
    assert(m_datahandshake == false);

    // is write response part of the channel?
    m_writeresp_enable=m_parameters.writeresp_enable;

    // is READ-EX part of the channel?
    m_readex_enable=m_parameters.readex_enable;

    // Are non-posted writes (write commands that receive responses) 
    //part of the channel?
    m_writenonpost_enable=m_parameters.writenonpost_enable;

    //-----------------------------------------
    //  Master Specific Parameters
    //-----------------------------------------

    // Retrieve any configuration parameters that were passed to this block
    // in the setConfiguration command.

#ifdef DEBUG
    cout << "I am configuring a Master!" << endl;
    cout << "Here is my configuration map for Master >" 
            << name() << "< that was passed to me." << endl;
    MapStringType::iterator map_it;
    for (map_it = m_parameters.Map.begin(); map_it != m_parameters.Map.end(); ++map_it) {
        cout << "map[" << map_it->first << "] = " << map_it->second << endl;
    }
    cout << endl;
#endif
    ipP->setOCPMasterConfiguration(m_parameters.Map);


    //in case sthreadbusy is part of the channel, this master will not be
    //default timing anymore and gets timing sensitive, too
    if (m_sthreadbusy){
      OCP_TL1_Master_TimingCl myTiming;
      //requests start after sthreadbusy is stable
      myTiming.RequestGrpStartTime=m_sthreadbusy_sample_time;
      ipP->registerTimingSensitiveOCPTL1Master((OCP_TL1_Slave_TimingIF*) this);
      ipP->setOCPTL1MasterTiming(myTiming);
    }
}

template<typename TdataCl>
Master<TdataCl>::~Master() {}

template<typename TdataCl>
void Master<TdataCl>::setOCPTL1SlaveTiming(OCP_TL1_Slave_TimingCl slave_timing){
  if (slave_timing.SThreadBusyStartTime+sc_time(1,SC_PS)>m_sthreadbusy_sample_time){
    m_sthreadbusy_sample_time=slave_timing.SThreadBusyStartTime+sc_time(1,SC_PS);
    OCP_TL1_Master_TimingCl myTiming;
    myTiming.RequestGrpStartTime=m_sthreadbusy_sample_time;
    ipP->setOCPTL1MasterTiming(myTiming);       
  }
}

template<typename TdataCl>
void Master<TdataCl>::requestThreadProcess()
{
    Ta Addr[] = {0x1784, 0x20, 0x20, 0x40};

    // start time of requests
    int NumWait[NUM_TESTS][4] = {
            {100,   3, 0xF, 0xF},
            {7,   1,   3, 0xF},
            {6, 0xF, 0xF, 0xF},
            {10,  2,   1, 0xF},
            {7,   1,   3, 0xF},
            {6,   1,   1,   1},
            {7,   2, 0xF, 0xF},
            {8,   2,   1, 0xF},// no data handshake
            {7,   2,   2,   2}
        };

    // specifies the command to use
    OCPMCmdType Commands[NUM_TESTS][4]  = {
        {OCP_MCMD_WR, OCP_MCMD_RD, OCP_MCMD_IDLE, OCP_MCMD_IDLE},
        {OCP_MCMD_WR, OCP_MCMD_WR, OCP_MCMD_WR, OCP_MCMD_IDLE},
        {OCP_MCMD_RD, OCP_MCMD_IDLE, OCP_MCMD_IDLE, OCP_MCMD_IDLE},
        {OCP_MCMD_RD, OCP_MCMD_RD, OCP_MCMD_RD, OCP_MCMD_IDLE},
        {OCP_MCMD_RD, OCP_MCMD_RD, OCP_MCMD_RD, OCP_MCMD_IDLE},
        {OCP_MCMD_RD, OCP_MCMD_RD, OCP_MCMD_RD, OCP_MCMD_RD},
        {OCP_MCMD_RD, OCP_MCMD_RD, OCP_MCMD_IDLE, OCP_MCMD_IDLE},
        {OCP_MCMD_WR, OCP_MCMD_WR, OCP_MCMD_WR, OCP_MCMD_IDLE},
        {OCP_MCMD_RD, OCP_MCMD_RD, OCP_MCMD_RD, OCP_MCMD_RD}
    };

    // number of specified transactions in a test
    int NumTr[] = {2, 3, 1, 3, 3, 4, 2, 3, 4};

    // -----------------------------------
    // (1) processing and preparation step
    // -----------------------------------

    // initialize data
    OCPRequestGrp<Td,Ta> req;
    int              Count = 0;
    int              Nr = 0;
    sc_time          old_time;
    sc_time          current_time;
    bool             sthreadbusy;
    unsigned int     my_data = 0;

    // calculate the new waiting time
    double  wait_for = NumWait[Nr][Count];

    // Do requests contain data (or will it be sent separately)
    // Always true as this core does not support data handshake
    req.HasMData = true;        

    wait();

    // main loop
    while (true) {
        // wait for the time to send the current request

        if (m_debug_os_ptr) {
            (*m_debug_os_ptr) << "DB (" << name() << "): "
                         << "master wait_for = " << wait_for << endl;
        }

        for (int i=0; i<wait_for; i++) wait();

        // remember the time
        old_time = sc_time_stamp();

        // ------------------------------------------------
        // (2) is SThreadBusy?
        // ------------------------------------------------
        
        // NOTE: we are single threaded so the thread busy signal
        // looks like a boolean (0 or 1). 
        //   Abritration based on thread busy will be needed for a
        //   multi-threaded model.
        if (m_sthreadbusy_exact) {
            wait(m_sthreadbusy_sample_time); //wait until sthreadbusy is stable
            sthreadbusy = ipP->getSThreadBusy();
            while (sthreadbusy) {
                wait();
                wait(m_sthreadbusy_sample_time); //wait until sthreadbusy is stable
                sthreadbusy = ipP->getSThreadBusy();
            }
        }

        // ------------------------------------------------
        // (3) send a request
        // ------------------------------------------------

        // NOTE: data handshake is not handled by this simple example.

        // Compute the next request
        req.MCmd = Commands[Nr][Count];

        // is this an extended command to be sent over a basic 
        // channel?
        if ( (!m_readex_enable) && (req.MCmd == OCP_MCMD_RDEX) ) {
            // channel cannot handle READ-EX. Send simple READ.
            req.MCmd = OCP_MCMD_RD;
        } else if ((!m_writenonpost_enable) && (req.MCmd == OCP_MCMD_WRNP)){
            // channel cannout handle WRITE-NP. Send simple WRITE.
            req.MCmd = OCP_MCMD_WR;
        }

        // compute the address
        req.MAddr = Addr[Count] + m_ID*0x40;
        req.MByteEn = 0xf;
        if (m_addrspace) {
            req.MAddrSpace = 0x1;
        }
        // compute the data
        switch (req.MCmd) {
            case OCP_MCMD_WR:
            case OCP_MCMD_WRNP:
            case OCP_MCMD_WRC:
            case OCP_MCMD_BCST:
                // This is a write command - it has data
                my_data++;
                // put the data into the request
                req.MData = my_data + m_ID*0x40;
                break;
            case OCP_MCMD_RD:
            case OCP_MCMD_RDEX:
            case OCP_MCMD_RDL:
                // this is a read command - no data.
                req.MData = 0;
                break;
            default:
                cout << "ERROR: Master \"" << name() 
                     << "\" generates unknown command #"
                          << req.MCmd << endl;
        }

        if (m_debug_os_ptr) {
            (*m_debug_os_ptr) << "DB (" << name() << "): "
                              << "send request." << endl;
            (*m_debug_os_ptr) << "DB (" << name() << "): "
                       << "    t = " << sc_simulation_time() << endl;
            (*m_debug_os_ptr) << "DB (" << name() << "): "
                              << "    MCmd: " << req.MCmd << endl;
            (*m_debug_os_ptr) << "DB (" << name() << "): "
                              << "    MData: " << req.MData << endl;
            (*m_debug_os_ptr) << "DB (" << name() << "): "
                           << "    MByteEn: " << req.MByteEn << endl;
        }

        // send the request
        ipP->startOCPRequestBlocking(req);
        // -------------------------------
        // (1) processing and preparation step
        // -------------------------------

        wait(); //advance to next cycle to avoid sending multiple request in same cycle

        // compute the next pointer
        if (++Count >= NumTr[Nr]) {
            Count = 0;
            if (++Nr >= NUM_TESTS) Nr = 1;
        }

        // calculate the new waiting time
        wait_for = NumWait[Nr][Count];
        current_time = sc_time_stamp();
        double delta_time = 
              (current_time.value() - old_time.value()) / 1000;
        if (delta_time >= wait_for) {
            wait_for = 0;
        } else {
            wait_for = wait_for - delta_time;
        }
    }
}

template<typename TdataCl>
void Master<TdataCl>::responseThreadProcess()
{

    // initialization
    OCPResponseGrp<Td> resp;
    double         wait_for;

    wait();

    // main loop
    while (true) {
        // ------------------------------------------------
        // (1) wait for a response (blocking wait)
        // ------------------------------------------------

        // get the next response
        ipP->getOCPResponseBlocking(resp);

        // ------------------------
        // (2) process the response
        // ------------------------

        // compute the response acceptance time
        if (m_respaccept_fixeddelay) {
           wait_for = m_respaccept_delay;
        } else {
           // Go random up to max delay
           wait_for = 
            (int)((m_respaccept_delay+1) * rand() / (RAND_MAX + 1.0));
        }

        // --------------------------------------------------
        // (3) generate a one-cycle-pulse MRespAccept signal
        // --------------------------------------------------

        if (m_respaccept) {
            if (wait_for == 0) {
                // send an one-cycle-pulse MRespAccept signal
                ipP->putMRespAccept();
            } else {
                // wait for the acceptance pulse cycle
                for (int i=0; i<wait_for; i++) wait();
                // send an one-cycle-pulse MRespAccept signal
                ipP->putMRespAccept();
            }
        }

        if (m_mthreadbusy_exact) {
           // use the MThreadBusy signal instead of resp accept
            if (wait_for > 0) {
                // Set MThreadBusy
                wait(); //wait until next cycle to set busy
                ipP->putMThreadBusy(1);
                // keep MThreadBusy on 
                for (int i=0; i<wait_for; i++) wait();
                // now release it
                ipP->putMThreadBusy(0);
            }
        }
    }
}

template<typename TdataCl>
void Master<TdataCl>::exerciseSidebandThreadProcess(void)
{
    // Systematically send out sideband signals on 
    // any signals that are attached to us.
    for (int i=0; i<10; i++) wait();
    int tweakCounter =0;
    bool hasMError=m_parameters.merror;
    bool nextMError = false;
    bool hasMFlag=m_parameters.mflag;
    int numMFlag=m_parameters.mflag_wdth;
    unsigned int nextMFlag = 0;
    unsigned int maxMFlag = (1 << numMFlag) -1; 

    // main loop
    while (true) {
        // wait 10 cycles
        for (int i=0; i<10; i++) wait();

        // Now count through my sideband changes
        tweakCounter++;

        // Drive MError
        if (hasMError) {
            if (tweakCounter%2 == 0) {
                // Toggle MERROR
                nextMError = !nextMError;
                ipP->MputMError(nextMError);
            }
        }

        // Drive MFlags
        if (hasMFlag) {
            if (tweakCounter%1 == 0) {
                // go to next MFlag
                nextMFlag += 1;
                if (nextMFlag > maxMFlag) {
                    nextMFlag = 0;
                }
                ipP->MputMFlag(nextMFlag);
            }
        }
    }
}

template class Master< OCP_TL1_DataCl<OCPCHANNELBit32, OCPCHANNELBit32> >;
