// 
//  (c) Copyright OCP-IP 2003, 2004
//  OCP-IP Confidential and Proprietary
//
// ============================================================================
//      Project : OCP SLD WG, Layer adapter examples 
//      Authors : Yann Bajot, PROSILOG, bajot@prosilog.com 
//
//  Description : Transaction Level - Layer-0 to Layer-1 Master Adapter (OCP 2.0)
//               (with Datahandshake support)
//  $Id: ocp2_tl0_tl1_master_adapter.cpp,v 1.5 2007/12/03 18:11:05 halexan Exp $
//
//     Features :
//
//    - Uses the OCP-specific TL1 Channel and associated API
//
//    - Layer adapter used to connect a TL0 OCP Master to a TL1 system
//
//    - Interface implements OCP Basic signals : Clk, MCmd, MAddr, MData,
//        SCmdAccept, SData, SResp & MRespAccept.
//        + Datahandshake signals: MDataValid, SDataAccept
//
//    - Glitches and propagation delays for input signals are removed using two
//        'Sample delay' parameters.  Signals coming from the TL0 master are
//        not supposed to be clocked and therefore can change at any time
//        during the clock cycle. It is also possible that some glitches appear
//        on them, particulary in the case of co-simulation with a
//        HDL-simulated RTL Master. 
//
//      Once a TL1 interface call has been done, it can not be cancelled. The
//      master adapter issues three kinds of call to the TL1 channel:
//          * startOCPRequest()
//          * startOCPDataHS()
//          * putMRespAccept() 
//
//      The startOCPRequest()' and associated API calls must happen only
//        when TL0 associated signals 'MCmd' and 'MAddr' are stable.  Similarly,
//        'startOCPDataHS()' call must wait 'MDataValid' and 'MData' to be
//        stable.  Since the propagation delays for these signals are master
//        dependant, we use a 'Request Sample Delay' parameter to determine when
//        these signals should be sampled during the cycle.
//
//      The 'putMRespAccept()' call must happen only when 'MRespAccept' is stable.
//        Since the propagation delay for this signal is master dependant, a
//        'RespAcceptSampleDelay' parameter is used to determine when
//        'MRespAccept' should be sampled during the cycle.
//
//      The smallest value for the sample delays is sc_get_time_resolution() ;
//      it should be used in the case when TL0 master has pure 'zero-delay'
//      signals. Users should always use a timed sample delay (and not a
//      defined number of delta-cycles) since it is not possible to predict the
//      number of delta-cycles required for signal stability.
//
//      'Check_setup_time()' function is proposed for debugging purposes, to
//        ensure that TL0 master output signals don't change after the sample
//        delays, which would cause the adapter to fail. A good way to proceed
//        is to perform the first simulation with checking function actived to
//        determine the appropriate sample delay value.  Once TL0 behaviour is
//        correct, checking function should be disabled to increase simulation
//        speed.
//                
// Parameters   : 
//   Template arguments.
//   - TdataCl: Data class containing data members and
//              access methods for the data exchanged between
//              Masters and Slaves
//   - Td: Data type
//   - Ta: Address type
//                
//    First Constructor arguments (with explicit sampling times):
//     - sc_module_name: Name of the module instance
//     - ID: Unique number identifying the Master.
//           ID must be unique among all Masters attached to the same Bus.
//     - Priority: Positive number specifying the priority relative to the
//                 other Masters for Bus access. Higher numbers means higher
//                 priority. Masters can have the same priority.
//     - RequestSampleDelay: Time number specifying the delay between clock 
//                 rising edge and the actual sampling time for Request signals
//                 and Datahandshake signals
//     - RespAcceptSampleDelay: Time number specifying the delay between clock 
//                 rising edge and the actual sampling time for 'MRespAccept'
//                 signal
//     - Check_setup_time: boolean value specifying if 'check_setup_time()' 
//                 function is executed during simulation 
//
//   Second Constructor arguments (sampling times use default values):
//     - sc_module_name: Name of the module instance
//     - ID: Unique number identifying the Master.
//           ID must be unique among all Masters attached to the same Bus.
//     - Priority: Positive number specifying the priority relative to the
//                 other Masters for Bus access. Higher numbers means higher
//                 priority. Masters can have the same priority.
//     - Combinational: Boolean specifying if DataHandshake/Response occurs in the
//                 same clock cycle or sequentially. Sampling times will be
//                 set as follows:
//                    - Combinational:
//                          * RequestSampleDelay = ClockCycle*60/100
//                          * ResponseSampleDelay = ClockCycle*75/100
//                    - Sequential:
//                          * RequestSampleDelay = ClockCycle*60/100
//                          * ResponseSampleDelay = ClockCycle*75/100
//     - ClockCycle: sc_time number specifying the clock cycle
//     - Check_setup_time: boolean value specifying if 'check_setup_time()' 
//                 function is executed during simulation 
//
//
//     NOTE : The adapter uses events of the communication class to
//     synchronize.  TL1 channel connected to the master port MUST have its
//     'syncevent' parameter set.
//
//     History:
//      06/10/2004: Added PE() support for TL1
// ============================================================================

#include "ocp2_tl0_tl1_master_adapter.h"
#include "ocp_tl_param_cl.h"
#include "ocp_tl1_data_cl.h"

// Used for debug purposes
//#define DEBUG_MASTER_ADAPTER

static
inline bool _isWriteCmd( const OCPMCmdType& cmd )
{
    return ( ( cmd == OCP_MCMD_WR   ) ||
             ( cmd == OCP_MCMD_WRNP ) ||
             ( cmd == OCP_MCMD_WRC  ) ||
             ( cmd == OCP_MCMD_BCST ) );
}

// ----------------------------------------------------------------------------
// Process : OCP2_TL0_TL1_Master_Adapter::OCP2_TL0_TL1_Master_Adapter
// 
// Constructor with explicit sampling timings
// ----------------------------------------------------------------------------
template<class TdataCl> OCP2_TL0_TL1_Master_Adapter<TdataCl>::OCP2_TL0_TL1_Master_Adapter
(
 sc_module_name name_
 , const OCP_TL0_PortFactory<OCP2_TL0_SlavePorts>& portFactory
 , bool checkSetupTime
 , bool combinational
    ) : sc_module         (name_)
        , MasterP                   ( "MasterPort"       ) 
        , Clk                       ( "clk"              )
        , m_tl0Ports                ( *( portFactory() ) )
        , m_check_active            ( checkSetupTime     ) 
        , m_default_values          ( false              ) 
        , m_response_in_progress    ( false              ) 
        , m_request_PE_mode         ( false              ) 
        , m_datahs_PE_mode          ( false              )
{
    // Init TL0 output signals:
    //    * 'SCmdAccept' as false
    //    * 'SDataAccept' as false
    //    * 'SResp' as OCP_SRESP_NULL
    //    * 'SData' as '0'
    if ( m_tl0Ports.SCmdAccept != NULL )
        m_tl0Ports.SCmdAccept->initValue( 0 );
    if ( m_tl0Ports.SDataAccept != NULL )
        m_tl0Ports.SDataAccept->initValue( 0 );
    if ( m_tl0Ports.SResp != NULL )
        m_tl0Ports.SResp->initValue( OCP_SRESP_NULL );
    if ( m_tl0Ports.SData != NULL )
        m_tl0Ports.SData->initValue( 0 );

    // Reset the ocp groups
    TL1_request.reset();
    TL1_dataHS.reset();

    // SC_METHODs
    SC_METHOD(send_Request);
    sensitive << e_request_sample_event;
    dont_initialize();

    if ( m_tl0Ports.MDataValid != NULL ) {
        SC_METHOD(send_Data);
        sensitive << e_request_sample_event;
    }

    SC_METHOD(set_SCmdAccept_Normal);
    sensitive << MasterP.RequestEndEvent();
    dont_initialize();

    SC_METHOD(set_reset_SCmdAccept_PE);
    sensitive << MasterP.RequestReleaseEvent();
    dont_initialize();

    if ( m_tl0Ports.SCmdAccept != NULL ) {                
        SC_METHOD(reset_SCmdAccept_Normal);
        sensitive_pos << Clk;
        dont_initialize();
    }

    SC_METHOD(set_reset_SDataAccept_PE);
    sensitive << MasterP.DataHSReleaseEvent();
    dont_initialize();
    
    SC_METHOD(set_SDataAccept_Normal);
    sensitive << MasterP.DataHSEndEvent();
    dont_initialize();

    if ( m_tl0Ports.SDataAccept != NULL ) {        
        SC_METHOD(reset_SDataAccept_Normal);
        sensitive_pos << Clk;
        dont_initialize();
    }

    if ( m_tl0Ports.SThreadBusy != NULL ) {
        SC_METHOD(set_TL0_SThreadBusy);
        sensitive << MasterP.CurrentSThreadBusyEvent();
        dont_initialize();
    }

    if ( m_tl0Ports.SDataThreadBusy != NULL ) {
        SC_METHOD(set_TL0_SDataThreadBusy);
        sensitive << MasterP.CurrentSDataThreadBusyEvent();
        dont_initialize();
    }

    SC_METHOD(set_TL1_ThreadBusy);
    sensitive << e_threadbusy_sample_event;
    dont_initialize();

    SC_METHOD(send_SReset);
    sensitive << MasterP.ResetStartEvent() << MasterP.ResetEndEvent();
    dont_initialize();

    if ( m_tl0Ports.MReset_n != NULL ) {
      SC_METHOD(send_MReset);
      sensitive << m_tl0Ports.MReset_n->defaultEvent();
    }

    SC_METHOD(reset_TL0_signals);
    sensitive_pos << Clk;

    SC_METHOD(send_Response);
    sensitive << MasterP.ResponseStartEvent();
    dont_initialize();

    SC_METHOD(release_Response);
    sensitive << e_respaccept_sample_event;

    SC_METHOD(sample_events_trigger);
    dont_initialize();
    sensitive_pos << Clk;

    if(m_check_active) {
        SC_METHOD(check_respaccept_setup_time);
        if ( m_tl0Ports.MRespAccept != NULL )
            sensitive << m_tl0Ports.MRespAccept->defaultEvent();
        SC_METHOD(check_request_setup_time);
        sensitive << m_tl0Ports.MCmd->defaultEvent();
        if ( m_tl0Ports.MAddr != NULL )
            sensitive << m_tl0Ports.MAddr->defaultEvent();
        if ( m_tl0Ports.MData != NULL )
            sensitive << m_tl0Ports.MData->defaultEvent();
        if ( m_tl0Ports.MDataValid != NULL )
            sensitive << m_tl0Ports.MDataValid->defaultEvent();

    }

    m_request_accepted = false;
    m_ready_for_request = true;
    m_datahs_accepted = false;
    m_ready_for_datahs = true;

}

// ----------------------------------------------------------------------------
//  Method : OCP2_TL0_TL1_Master_Adapter::setClockPeriod(), setSampleDelays
//
//  Set up the accept/response sample delays relative to the clock cycle
//  or as absolute delays.
//  This is separate from construction because clock period may not be known
//  if embedded in a structural module. However, this must be called before
//  elaboration
//  (when all binding operations are finished).
// ----------------------------------------------------------------------------
template<class TdataCl>
void OCP2_TL0_TL1_Master_Adapter<TdataCl>::setClockPeriod( const sc_time& period )
{    
    // fractions of cycle loosely based on Level2 timing guidelines
    m_request_sample_delay    = sc_time( period * double( 0.50 ) );
    m_respaccept_sample_delay = sc_time( period * double( 0.75 ) );
    m_threadbusy_sample_delay = sc_time( period * double( 0.10 ) );
}  

template<class TdataCl>
void OCP2_TL0_TL1_Master_Adapter<TdataCl>::setSampleDelays( const sc_time& requestDelay,
                                                            const sc_time& respAcceptDelay,
                                                            const sc_time& threadBusyDelay )
{
    m_request_sample_delay    = requestDelay;
    m_respaccept_sample_delay = respAcceptDelay;
    m_threadbusy_sample_delay = threadBusyDelay;
}

// ----------------------------------------------------------------------------
// Process : OCP2_TL0_TL1_Master_Adapter::~OCP2_TL0_TL1_Master_Adapter
// 
// Destructor
// ----------------------------------------------------------------------------
template<class TdataCl> OCP2_TL0_TL1_Master_Adapter<TdataCl>::~OCP2_TL0_TL1_Master_Adapter()
{
}

// ----------------------------------------------------------------------------
//  Method : OCP2_TL0_TL1_Slave_Adapter::send_SReset()
//  Pass SReset from TL1 to TL0
//
//  Sensitivity: 
//  sensitive << SlaveP.ResetStartEvent() << SlaveP.ResetEndEvent()
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP2_TL0_TL1_Master_Adapter<TdataCl>::send_SReset()
{
    bool activeLowResetValue = !( MasterP->getReset() );
    if ( m_tl0Ports.SReset_n != NULL )
        m_tl0Ports.SReset_n->putValue( activeLowResetValue );
}

// ----------------------------------------------------------------------------
//  Method : OCP2_TL0_TL1_Slave_Adapter::send_MReset()
//  Pass MReset from TL0 to TL1
//
//  Sensitivity: 
//  sensitive << m_tl0Ports.MReset_n->default_event();
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP2_TL0_TL1_Master_Adapter<TdataCl>::send_MReset()
{
    bool inReset = !( m_tl0Ports.MReset_n->getValue() );
    if ( inReset )
        MasterP->MResetAssert();
    else
        MasterP->MResetDeassert();
}

// ----------------------------------------------------------------------------
//  Method : OCP2_TL0_TL1_Master_Adapter::send_Request()
//
//  Catch the TL0 requests and send them to the TL1 channel 
//
//  Sensitivity: 
//      sensitive << e_request_sample_event;
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP2_TL0_TL1_Master_Adapter<TdataCl>::send_Request()
{
    OCPMCmdType mCmd = static_cast<OCPMCmdType>( m_tl0Ports.MCmd->getValue() );
    unsigned int threadId = 0;
    if ( m_tl0Ports.MThreadID != NULL )
        threadId = m_tl0Ports.MThreadID->getValue();
    TdataCl* pDataCl = MasterP->GetDataCl();
    unsigned int sThreadBusyVec = 0;
    if ( m_ParamCl->sthreadbusy )
        sThreadBusyVec = MasterP->getSThreadBusy();
    if( m_ready_for_request   && !MasterP->getSBusy() && mCmd != OCP_MCMD_IDLE &&
        !pDataCl->isThisThreadBusy( threadId, sThreadBusyVec ) ) {
        // This is a new request. No more requests sent until the current one
        // has been accepted
        m_ready_for_request = false;
        TL1_request.MCmd  = mCmd;
        TL1_request.MThreadID = threadId;
        if ( m_ParamCl->addr )
            m_tl0Ports.MAddr->getLongValue( TL1_request.MAddr );
        if ( m_ParamCl->addrspace )
            TL1_request.MAddrSpace = m_tl0Ports.MAddrSpace->getValue();
        if ( m_ParamCl->byteen )
            TL1_request.MByteEn = m_tl0Ports.MByteEn->getValue();
        if ( m_ParamCl->connid )
            TL1_request.MConnID = m_tl0Ports.MConnID->getValue();
        if ( m_ParamCl->reqinfo )
            m_tl0Ports.MReqInfo->getLongValue( TL1_request.MReqInfo );
        if ( m_ParamCl->atomiclength )
            TL1_request.MAtomicLength = m_tl0Ports.MAtomicLength->getValue();
        if ( m_ParamCl->burstlength )
            TL1_request.MBurstLength = m_tl0Ports.MBurstLength->getValue();
        if ( m_ParamCl->blockheight )
            TL1_request.MBlockHeight = m_tl0Ports.MBlockHeight->getValue();
        if ( m_ParamCl->blockstride )
            TL1_request.MBlockStride = m_tl0Ports.MBlockStride->getValue();
        if ( m_ParamCl->burstseq ) {
            OCPMBurstSeqType seq = static_cast<OCPMBurstSeqType>(
                m_tl0Ports.MBurstSeq->getValue() );
            TL1_request.MBurstSeq = seq;
        }
        if ( m_ParamCl->burstprecise )
            TL1_request.MBurstPrecise = m_tl0Ports.MBurstPrecise->getValue();
        if ( m_ParamCl->burstsinglereq )
            TL1_request.MBurstSingleReq = m_tl0Ports.MBurstSingleReq->getValue();
        if ( m_ParamCl->reqlast )
            TL1_request.MReqLast = m_tl0Ports.MReqLast->getValue();
        if ( m_ParamCl->reqrowlast )
            TL1_request.MReqRowLast = m_tl0Ports.MReqRowLast->getValue();

        if ( _isWriteCmd( TL1_request.MCmd ) && !m_ParamCl->datahandshake ) {
            TL1_request.HasMData = true;
            m_tl0Ports.MData->getLongValue( TL1_request.MData );
            if ( m_ParamCl->mdatainfo )
                m_tl0Ports.MDataInfo->getLongValue( TL1_request.MDataInfo );
        }
        if ( m_ParamCl->tags > 1 ) {
            TL1_request.MTagID = m_tl0Ports.MTagID->getValue();
            if ( m_ParamCl->taginorder )
                TL1_request.MTagInOrder = m_tl0Ports.MTagInOrder->getValue();
        }

        bool success = MasterP->startOCPRequest(TL1_request);
        assert( success );
    }
} // end of SC_METHOD

// ----------------------------------------------------------------------------
//  Method : OCP2_TL0_TL1_Master_Adapter::send_Data()
//
//  Send data requests to TL1 channel (handshake support)
//
//  Sensitivity: 
//      sensitive << e_request_sample_event;
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP2_TL0_TL1_Master_Adapter<TdataCl>::send_Data()
{
    bool dataValid = m_tl0Ports.MDataValid->getValue();
    if( m_ready_for_datahs && !MasterP->getSBusyDataHS() && dataValid ) {

        // This is a new request. No more requests sent until the current one
        // has been accepted
        m_ready_for_datahs = false;

        m_tl0Ports.MData->getLongValue( TL1_dataHS.MData );
        if ( m_ParamCl->threads > 1 )
            TL1_dataHS.MDataThreadID = m_tl0Ports.MDataThreadID->getValue();
        if ( m_ParamCl->mdatainfo )
            m_tl0Ports.MDataInfo->getLongValue( TL1_dataHS.MDataInfo );
        if ( m_ParamCl->datalast )
            TL1_dataHS.MDataLast = m_tl0Ports.MDataLast->getValue();
        if ( m_ParamCl->datarowlast )
            TL1_dataHS.MDataRowLast = m_tl0Ports.MDataRowLast->getValue();
        if ( m_ParamCl->mdatabyteen )
            TL1_dataHS.MDataByteEn = m_tl0Ports.MDataByteEn->getValue();
        if ( m_ParamCl->tags > 1 )
            TL1_dataHS.MDataTagID = m_tl0Ports.MDataTagID->getValue();

        MasterP->startOCPDataHS(TL1_dataHS);
    }
} // end of SC_METHOD

// ----------------------------------------------------------------------------
//  Method : OCP2_TL0_TL1_Master_Adapter::set_SCmdAccept_Normal()
//
//  Set 'SCmdAccept' signal in normal mode
//
//  Sensitivity: 
//      sensitive << MasterP.RequestEndEvent();
// ----------------------------------------------------------------------------
template<class TdataCl> void
OCP2_TL0_TL1_Master_Adapter<TdataCl>::set_SCmdAccept_Normal()
{
    // The current TL1 request has been accepted    
    m_request_accepted = true;
    if ( m_tl0Ports.SCmdAccept != NULL ) 
        m_tl0Ports.SCmdAccept->putValue( 1 );
} // end of SC_METHOD


// ----------------------------------------------------------------------------
//  Method : OCP2_TL0_TL1_Master_Adapter::reset_SCmdAccept_Normal()
//
//  Reset 'SCmdAccept' TL0 signal in normal mode 
//
//  Sensitivity: 
//      sensitive_pos << Clk
// ----------------------------------------------------------------------------
template<class TdataCl> void
OCP2_TL0_TL1_Master_Adapter<TdataCl>::reset_SCmdAccept_Normal()
{
    if(!m_request_PE_mode) 
        m_tl0Ports.SCmdAccept->putValue( 0 );
} // end of SC_METHOD

// ----------------------------------------------------------------------------
//  Method : OCP2_TL0_TL1_Master_Adapter::set_reset_SCmdAccept_PE()
//
//  Set and reset 'SCmdAccept' TL0 signal in PE mode
//
//  Sensitivity: 
//      sensitive << MasterP.RequestReleaseEvent()
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP2_TL0_TL1_Master_Adapter<TdataCl>::set_reset_SCmdAccept_PE()
{

    if(m_CommCl->RequestRelease) {
        // Entering the PE mode
        m_request_PE_mode = true;
        // TL1 slave called putSCmdAccept(true)
        if ( m_tl0Ports.SCmdAccept != NULL ) 
            m_tl0Ports.SCmdAccept->putValue( 1 );
        // The current TL1 request has been accepted
        m_request_accepted = true;
    } else {
        // Leaving the PE mode
        m_request_PE_mode = false;
        // TL1 slave called putSCmdAccept(false)
        if ( m_tl0Ports.SCmdAccept != NULL ) 
            m_tl0Ports.SCmdAccept->putValue( 0 );
    }

} // end of SC_METHOD

// ----------------------------------------------------------------------------
//  Method : OCP2_TL0_TL1_Master_Adapter::set_SDataAccept_Normal()
//
//  Set 'SDataAccept' signal in normal mode
//
//  Sensitivity: 
//      sensitive << MasterP.DataRequestEndEvent();
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP2_TL0_TL1_Master_Adapter<TdataCl>::set_SDataAccept_Normal()
{
    // The current TL1 dataHS has been accepted
    m_datahs_accepted = true;
    if ( m_tl0Ports.SDataAccept != NULL )
        m_tl0Ports.SDataAccept->putValue( 1 );
} // end of SC_METHOD

// ----------------------------------------------------------------------------
//  Method : OCP2_TL0_TL1_Master_Adapter::reset_SDataAccept_Normal()
//
//  Reset 'SDataAccept' TL0 signal in normal mode 
//
//  Sensitivity: 
//      sensitive_pos << Clk
// ----------------------------------------------------------------------------
template<class TdataCl> void
OCP2_TL0_TL1_Master_Adapter<TdataCl>::reset_SDataAccept_Normal()
{
    if(!m_datahs_PE_mode)
        m_tl0Ports.SDataAccept->putValue( 0 );
} // end of SC_METHOD

// ----------------------------------------------------------------------------
//  Method : OCP2_TL0_TL1_Master_Adapter::set_reset_SDataAccept_PE()
//
//  Set and reset 'SDataAccept' TL0 signal in PE mode
//
//  Sensitivity: 
//      sensitive << MasterP.DataHSReleaseEvent()
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP2_TL0_TL1_Master_Adapter<TdataCl>::set_reset_SDataAccept_PE()
{

    if(m_CommCl->DataRelease) {
        // Entering the PE mode
        m_datahs_PE_mode = true;
        // TL1 slave called putSDataAccept(true)
        if ( m_tl0Ports.SDataAccept != NULL ) 
            m_tl0Ports.SDataAccept->putValue( 1 );
        // The current TL1 dataHS has been accepted
        m_datahs_accepted = true;
    } else {
        // Leaving the PE mode
        m_datahs_PE_mode = false;
        // TL1 slave called putSDataAccept(false)
        if ( m_tl0Ports.SDataAccept != NULL  )
            m_tl0Ports.SDataAccept->putValue( 0 );
    }

} // end of SC_METHOD

// ----------------------------------------------------------------------------
//  Method : OCP2_TL0_TL1_Master_Adapter::set_TL0_SThreadBusy/SDataThreadBusy
//
//  forward threadbusy signals
//
//  Sensitivity: 
//      sensitive_pos << MasterP.SThreadBusyEvent()
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP2_TL0_TL1_Master_Adapter<TdataCl>::set_TL0_SThreadBusy()
{
    if ( m_tl0Ports.SThreadBusy != NULL  )
        m_tl0Ports.SThreadBusy->putValue( MasterP->getCurrentSThreadBusy() );
}
template<class TdataCl> void OCP2_TL0_TL1_Master_Adapter<TdataCl>::set_TL0_SDataThreadBusy()
{
    if ( m_tl0Ports.SDataThreadBusy != NULL  )
        m_tl0Ports.SDataThreadBusy->putValue( MasterP->getCurrentSDataThreadBusy() );
}

// ----------------------------------------------------------------------------
//  Method : OCP2_TL0_TL1_Master_Adapter::set_TL1_ThreadBusy()
//
//  Set TL1 threadbusy state after sampling TL0 MThreadBusy signal
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP2_TL0_TL1_Master_Adapter<TdataCl>::set_TL1_ThreadBusy()
{
    if ( m_tl0Ports.MThreadBusy != NULL ) {
        MasterP->putMThreadBusy( m_tl0Ports.MThreadBusy->getValue() );
    }
}


// ----------------------------------------------------------------------------
//  Method : OCP2_TL0_TL1_Master_Adapter::reset_TL0_signals()
//
//  Reset Response group TL0 signals 
//
//  Sensitivity: 
//      sensitive_pos << Clk;
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP2_TL0_TL1_Master_Adapter<TdataCl>::reset_TL0_signals()
{
    if( !m_response_in_progress && m_tl0Ports.SResp != NULL ) {
        m_tl0Ports.SResp->putValue( OCP_SRESP_NULL );
    }
    // The current TL1 request has been accepted by the slave, and this is the
    // end of the OCP cycle => return to 'IDLE' state
    OCPMCmdType mCmd = static_cast<OCPMCmdType>( m_tl0Ports.MCmd->getValue() );
    if ( mCmd != OCP_MCMD_IDLE && m_request_accepted) {
        m_ready_for_request = true;
        m_request_accepted = false;
    }

    // The current TL1 dataHS has been accepted by the slave, and this is the
    // end of the OCP cycle => return to 'IDLE' state
    bool dataValid = true;
    if ( m_tl0Ports.MDataValid != NULL )
        dataValid = m_tl0Ports.MDataValid->getValue();
    if ( dataValid && m_datahs_accepted ) {
        m_ready_for_datahs = true;
        m_datahs_accepted = false;
    }
} // end of SC_METHOD


// ----------------------------------------------------------------------------
//  Method : OCP2_TL0_TL1_Master_Adapter::send_Response()
//
//  Catch responses from the TL1 Slave and send them to the TL0 master 
//
//  Sensitivity: 
//      sensitive << MasterP.ResponseStartEvent();
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP2_TL0_TL1_Master_Adapter<TdataCl>::send_Response()
{
    if(MasterP->getOCPResponse(TL1_response)) {
        m_response_in_progress = m_ParamCl->respaccept;
        m_tl0Ports.SResp->putValue( TL1_response.SResp );
        if ( m_ParamCl->threads > 1 )
            m_tl0Ports.SThreadID->putValue( TL1_response.SThreadID );
        if ( m_ParamCl->sdata )
            m_tl0Ports.SData->putValue( TL1_response.SData );
        if ( m_ParamCl->sdatainfo )
            m_tl0Ports.SDataInfo->putValue( TL1_response.SDataInfo );
        if ( m_ParamCl->respinfo )
            m_tl0Ports.SRespInfo->putValue( TL1_response.SRespInfo );
        if ( m_ParamCl->resplast )
            m_tl0Ports.SRespLast->putValue( TL1_response.SRespLast );
        if ( m_ParamCl->resprowlast )
            m_tl0Ports.SRespRowLast->putValue( TL1_response.SRespRowLast );
        if ( m_ParamCl->tags > 1 )
            m_tl0Ports.STagID->putValue( TL1_response.STagID );
    }
    
#ifdef DEBUG_MASTER_ADAPTER
    cout << "TL0 Master Adapter sent response"
         << " delta " << simcontext()->delta_count()
         << " time " << sc_time_stamp().to_default_time_units()
         << " SResp " << TL1_response.SResp
         << " SData " << TL1_response.SData
         << " SThreadID " << TL1_response.SThreadID
         << endl;
#endif

} // end of SC_METHOD 

// ----------------------------------------------------------------------------
//  Method : OCP2_TL0_TL1_Master_Adapter::release_Response()
//
//  Release response present on TL1 channel
//
//  Sensitivity: 
//      sensitive << e_respaccept_sample_event;
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP2_TL0_TL1_Master_Adapter<TdataCl>::release_Response()
{
    if(m_response_in_progress) {
        bool respAccept = true;
        if ( m_tl0Ports.MRespAccept != NULL )
            respAccept = m_tl0Ports.MRespAccept->getValue();
        if( respAccept ) {
#ifdef DEBUG_MASTER_ADAPTER
            cout << "TL0-TL1 Master:"
            << "\tdelta " << simcontext()->delta_count()
            << "\ttime " << sc_time_stamp().to_default_time_units()
            << "\tcalls putMRespAccept()" << sc_time_stamp().to_default_time_units()
            << endl;
#endif       
            MasterP->putMRespAccept(); 
            m_response_in_progress = false;
        }
    }
} // end of SC_METHOD 


// ----------------------------------------------------------------------------
//  Method : OCP2_TL0_TL1_Master_Adapter::sample_events_trigger()
//
//  Trigger 'e_respaccept_sample_event' and 'e_request_sample_event'
//  used to sample input signals 
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP2_TL0_TL1_Master_Adapter<TdataCl>::sample_events_trigger()
{
#ifdef DEBUG_MASTER_ADAPTER
    cout << "TL0-TL1 Master:"
         << "\tdelta " << simcontext()->delta_count()
         << "\ttime " << sc_time_stamp().to_default_time_units()
         << "\tOCP Clock cycle"
         << endl;
#endif

    if ( m_ParamCl->respaccept )
        e_respaccept_sample_event.notify(m_respaccept_sample_delay);
    e_request_sample_event.notify(m_request_sample_delay);
    if ( m_ParamCl->mthreadbusy )
        e_threadbusy_sample_event.notify(m_threadbusy_sample_delay);

    if(m_check_active)
        m_last_rising_edge = sc_time_stamp();

} // end of SC_METHOD 

// ----------------------------------------------------------------------------
//  Method : OCP2_TL0_TL1_Master_Adapter::check_request_setup_time()
//
//  Checks timing violations, i.e. if 'MCmd', 'MAddr', 'MData' or 'MDataValid'
//  signals change after the sample time
//    
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP2_TL0_TL1_Master_Adapter<TdataCl>::check_request_setup_time()
{
    if( (sc_time_stamp()-m_last_rising_edge) >= m_request_sample_delay ) {
        cout << "TL0 Master Adapter '" << name() << "': Setup Time Violation "
            << " delta " << simcontext()->delta_count()
            << " time " << sc_time_stamp().to_default_time_units() << endl
            << " last rising edge " << m_last_rising_edge.to_default_time_units() << endl
            << " one of the signal 'MCmd', 'MData' , 'MAddr' or 'MDataValid' had a late change" << endl
            << " You have to adjust 'RequestSampleDelay' parameter" 
            << endl;

        if(m_default_values) {
            cout << " Default values for sampling times do not work with that design." << endl
                << " You have to explicitly specify sampling times using the appropriate constructor" 
                << endl;
        }

        sc_stop();
    }

} // end of SC_METHOD 

// ----------------------------------------------------------------------------
//  Method : OCP2_TL0_TL1_Master_Adapter::check_respaccept_setup_time()
//
//  Checks timing violations, i.e. if 'MRespAccept' signal
//    change after the sample time
//    
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP2_TL0_TL1_Master_Adapter<TdataCl>::check_respaccept_setup_time()
{
    if( (sc_time_stamp()-m_last_rising_edge) >= m_respaccept_sample_delay ) {
        cout << "TL0 Master Adapter: '" << name() << "' Setup Time Violation "
            << " delta " << simcontext()->delta_count()
            << " time " << sc_time_stamp().to_default_time_units() << endl
            << " last rising edge " << m_last_rising_edge.to_default_time_units() << endl
            << " signal 'MRespAccept' had a late change" << endl
            << " You have to adjust 'RespAcceptSampleDelay' parameter" 
            << endl;

        if(m_default_values) {
            cout << " Default values for sampling times do not work for that design." << endl
                << " You have to explicitly specify sampling times using the appropriate constructor" 
                << endl;
        }

        sc_stop();
    }

} // end of SC_METHOD 



// ----------------------------------------------------------------------------
//  Method : OCP2_TL0_TL1_Master_Adapter::SputDirect()
//
//  Debug interface method.
//  Read/Write data, Slave to Master direction
//  Returns true for success and false for failure
//
// ----------------------------------------------------------------------------
template<class TdataCl> bool OCP2_TL0_TL1_Master_Adapter<TdataCl>::SputDirect(
        int SlaveID, bool IsWrite, Td *Data, Ta Address, int NumWords)

{
    // not implemented
    return(false);
}


// ----------------------------------------------------------------------------
//  Method : OCP2_TL0_TL1_Master_Adapter::end_of_elaboration()
//
//  This method is activated at the end of the elaboration phase
//  (when all binding operations are finished).
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP2_TL0_TL1_Master_Adapter<TdataCl>::end_of_elaboration()
{
    sc_module::end_of_elaboration();

    // Get communication structure
    m_CommCl = MasterP->GetCommCl();
    
    // Here we check that the TL1 channel has 'cmdaccept' and 'dataaccept'
    m_ParamCl = MasterP->GetParamCl();

    if(m_request_sample_delay < sc_get_time_resolution() ) {
        cout << "Error: TL0 OCP Master Adapter (instance '" << name() << "')"<< endl
             << "       'Request Sample Delay' parameter must be > 'sc_get_time_resolution()' " << endl 
             << "       'Request Sample Delay' =" << m_request_sample_delay
             << endl;
        exit(-1);
    }
    
    if(m_respaccept_sample_delay < sc_get_time_resolution() ) {
        cout << "Error: TL0 OCP MasterAdapter (instance '" << name() << "')"<< endl
             << "       'RespAccept Sample Delay' parameter must be > 'sc_get_time_resolution()' " << endl 
             << "       'RespAccept Sample Delay' =" << m_respaccept_sample_delay
             << endl;
        exit(-1);
    }

    if(m_threadbusy_sample_delay < sc_get_time_resolution() ) {
        cout << "Error: TL0 OCP MasterAdapter (instance '" << name() << "')"<< endl
             << "       'ThreadBusy Sample Delay' parameter must be > 'sc_get_time_resolution()' " << endl 
             << "       'ThreadBusy Sample Delay' =" << m_threadbusy_sample_delay
             << endl;
        exit(-1);
    }

    // Initialize the current burst sequence type for each thread to the default
    m_curBurstSeq.clear();
    for (int i=0; i < (m_ParamCl->threads); ++i) {
        m_curBurstSeq.push_back(OCP_MBURSTSEQ_INCR);
    }
    
}
