// 
//  Copyright 2004 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG, Layer adapter examples 
//      Authors : Stephane Guntz, PROSILOG, guntz@prosilog.com
//				  Yann Bajot, PROSILOG, bajot@prosilog.com                
//         Date : 01/05/2004
//
//  Description : Transaction Level - Layer-1 to Layer-2 Slave Adapter (OCP 2.0 supported) 
//  Version: 1.0
//    
//  Parameters    :
//    Template arguments.
//     - TdataCl_tl1: TL1 Data class containing data members and
//                access methods for the data exchanged between
//                Masters and Slaves
//	   - TdataCl_tl2: TL2 Data class containing data members and
//                access methods for the data exchanged between
//                Masters and Slaves
//     - Td: Data type 
//     - Ta: Address type
//                
//	Constructor arguments:
//      - sc_module_name: Name of the module instance
//	    - max_chunk_length: maximum size of a response chunk
//		- max_burst_length: Maximum size for a burst
//		- adapter_depth: Number of missed events and responses to be cached.
//						 The adapter stores new request/response infos which have arrived during the processing of the current request/response.
//						 It sends them when the current request/response is finished.
//	 
//  The TL1 slave adapter has a TL2 slave interface on one side, and a TL1 master interface on the other side.
//  It must be connected to:
//      - a TL2 master through a TL2 channel
//      - a TL1 slave through a TL1 channel
//
//	The adapter retrieves TL2 requests and stores them in its internal buffer. It sends corresponding TL1 requests to the TL1 slave.
//	  
//  On the response side, the TL1 slave adapter retrieves the TL1 responses from the slave, packs them in burst and sends corresponding
//  TL2 responses to the master. 
//
//	The following is unsupported:
//	- Single request/multiple data
//	- Data handshake
//	- Multiple threads
//	- SThreadBusy/MThreadBusy compliances (m_threadbusy_exact=0, s_threadbusy_exact=0)
//	- address sequence which is not STRM or INCR
// 
// ============================================================================

#ifndef _OCP_TL1_TL2_SLAVE_ADAPTER_H
#define _OCP_TL1_TL2_SLAVE_ADAPTER_H

#include "systemc.h"
#include "ocp_globals.h"
#include "ocp_tl_param_cl.h"
#include "tl_slave_if.h"
#include "tl_master_if.h"
#include "ocp_tl1_data_cl.h"
#include "ocp_tl2_data_cl.h"
#include "ocp_tl2_slave_port.h"
#include "ocp_tl1_master_port.h"

//for debugging purposes, print messages on screen
//comment for no messages
#define DEBUG_SLAVE_ADAPTER_TL1

#ifndef SLAVE_ADAPTER_TASK_ENUM
#define SLAVE_ADAPTER_TASK_ENUM

enum SlaveAdapterTask   {
	SEND_CHUNK_WITH_DATA = 0,
	SEND_CHUNK_WITHOUT_DATA,
	STORE_NEW_DATA,
    SEND_CHUNK_AND_SINGLE_RESPONSE
};

#endif

template <class TdataCl_tl1, class TdataCl_tl2> class OCP_TL1_TL2_Slave_Adapter
  : public sc_module
   , public MdirectIF<TdataCl_tl2>
   , public SdirectIF<TdataCl_tl1>
{
public:  
  //TL1 data type and address type
  typedef typename TdataCl_tl1::DataType Td;
  typedef typename TdataCl_tl1::AddrType Ta;


  // TL2 Slave Port and TL1 Master port
  OCP_TL1_MasterPort< TdataCl_tl1> MasterP;
  OCP_TL2_SlavePort< int, int> SlaveP;
	  
  //clock port
  sc_in_clk clk;

  SC_HAS_PROCESS(OCP_TL1_TL2_Slave_Adapter);

  // constructor
  OCP_TL1_TL2_Slave_Adapter(sc_module_name name, int max_chunk_length, int max_burst_length, int adapter_depth);

  // destructor
  ~OCP_TL1_TL2_Slave_Adapter();

  // SystemC processes
  void MasterRequest();
  void SlaveRequest();
  void MasterResponse();
  void SlaveResponse();


  //direct interfaces methods
  virtual bool MputDirect(int, bool, Td*, Ta, int);
  virtual bool SputDirect(int, bool, Td*, Ta, int);

  //adapter methods
  bool same_response_fields(OCPResponseGrp<Td>&, OCPResponseGrp<Td>&);
  SlaveAdapterTask define_adapter_task(void);

 private :
  void end_of_elaboration();

  // pointer to the TL1 and TL2 data structures of the channel
  TdataCl_tl1          *m_DataCl_tl1;
  TdataCl_tl2          *m_DataCl_tl2;
  
  //pointer to param class
  ParamCl<TdataCl_tl1> *m_ParamCl;

  //pointer to communication class
  CommCl           *m_CommCl;

  // parameters given as constructor parameters
  int max_chunk_length;			  //maximum length of a response chunk
  unsigned int max_burst_length;		  //maximum size for a burst
  unsigned int buffer_size;            //size of the internal buffer for data and request fields (MCmd, MAddr, MAddrSpace...)
  int adapter_depth;		  //depth of requests and response storage 

  //internal parameters
  bool new_chunk_transaction;  //indicates if the next response is the beginning of a new chunk
  int m_NumBytesPerWord;	   // size of the data
  int* nb_response;				// number of TL1 responses that the slave sends
  int count_response;
  bool burst_support;			//does the channel has burst support or not
  bool burst_precise;			//bursts are precise or not
  bool write_response_enable;
 

  //indicates a later release of the request or response thread, when the internal buffer for storing request/response is full
  bool release_request_thread; 
  bool release_response_thread; 
  
  //index of the currently processed request/response
  int current_processed_request;
  int current_processed_response;
  
  unsigned int beginning_burst_index, current_index;   //index for internal buffers
  
  //buffers for storing beginning index and length of request/response
  int* burst_response_index;		//beginning index of the TL2 response
  unsigned int* burst_response_length;		//length of the TL2 response
  int* request_index;	//beginning index of the group of TL1 requests
  unsigned int* request_length;	//number of TL1 requests to send
  bool* last_of_a_burst;
  bool* chunk_is_last_of_burst;

  int response_length;
  bool chunk_response_last;
  int current_response_burst;
  int response_index;

  int m_pending_request;   //indicates if there is a pending request which has arrived, during the processing of the current request
  int m_request_number;	   //ID of the request currently processed
  int m_pending_response;  //indicates if there is a pending response which has arrived, during the processing of the current request
  int m_response_number;   //ID of the response currently processed

  //internal buffers
  OCPResponseGrp<Td>* m_response_buffer;  //for storing TL1 response fields
  OCPRequestGrp<Td, Ta>* m_request_buffer;    //for storing TL2 request fields
  Td* m_data;							//for storing data
  Td* m_response_data;					//for storing response
  
  //events
  sc_event send_TL1_request_event;	//event for sending the group of TL1 requests, made from the TL2 request
  sc_event send_TL2_response_event; //event for sending TL2 response, made of TL1 atomic responses
  sc_event thread_released_event; //event indicating the release of the response thread
  
};


#endif // _OCP_TL1_TL2_SLAVE_ADAPTER_H
