// Copyright (c) 2004 Sonics, Inc.
//
// Confidential and Proprietary Information of Sonics, Inc.
// Use, disclosure, or reproduction is prohibited without
// written permission from Sonics, Inc.
//
// $Id: StlCommand.cc,v 1.1 2007/01/25 22:09:07 halexan Exp $

#include "StlCommand.h"
#include "ocp_utils.h"
#include <sstream>
#include <cmath>

using namespace std;
using namespace OcpIp;

static NamedIndex<OcpIp::SignalName> s_ocp2Names;
//////////////////////////////////////////////////////////////////
// FUNCTION: StlParserSignalCommand::StlParserSignalCommand
// DESCRIPTION: Constructor:
// ARGUMENTS: ParamSet<OcpIp::ParamName>
//////////////////////////////////////////////////////////////////
StlParserSignalCommand::StlParserSignalCommand( const OcpParams& params ) :
    StlSignalCommand(),
    m_ocpParams( params )
{}

//////////////////////////////////////////////////////////////////
// FUNCTION: StlParserSignalCommand::drivenValue
// DESCRIPTION: Gets the value to be driven on the given signal
// ARGUMENTS: SignalName, value , mask (for flag)
// RETURNS: true if the command wats to drive the signal, false otherwise
// PRECONDITIONS: getSignal must have been called to determine
// the proper drivenValue call to make.
//////////////////////////////////////////////////////////////////
SignalName
StlSignalCommand::getSignal( bool& tbflags_ ) const {
    tbflags_ = tbflags;
    if ( tbflags ) return static_cast<SignalName>( 0 );
    else           return signal;
}

bool
StlSignalCommand::drivenValue() const
{
    assert( signal != S_MFlag && !tbflags );
    return (bool)( value & 1 );
}

snx_uint64_t
StlSignalCommand::drivenValue( snx_uint64_t& mask_ ) const
{
    assert( signal == S_MFlag || tbflags );
    mask_ = mask;
    return value;
}

void
StlParserSignalCommand::clear()
{
    _tbflags() = false;
    _value() = _mask() = static_cast<snx_uint64_t>(-1);
}

int
StlParserSignalCommand::reportError( string& errors ) const
{
    if ( tbflags ) {
        // no way to verify this
        return false;
    }

    const snx_uint64_t maxFlagVal = ( 1ULL << ocpParam( P_mflag_wdth ) ) -1;
    uint32_t numErrors( 0 );
    bool exists = false;
    switch ( signal ) {
    case S_MError     : exists = ocpParam( P_merror )     ; break;
    case S_ControlBusy: exists = ocpParam( P_controlbusy ); break;
    case S_StatusBusy : exists = ocpParam( P_statusbusy ) ; break;
    case S_MFlag      : exists = ocpParam( P_mflag )      ; break;
    default           : assert( false );
    }
    ostringstream errorStr;
    if (!exists) {
        numErrors++;
        errorStr << "signal " << s_ocp2Names.getName( signal )
                 << " is not enabled" << "\n";
    } else if ( signal == S_MFlag &&
                value != static_cast<snx_uint64_t>(-1) &&
                value > maxFlagVal ) {
        numErrors++;
        errorStr << "signal value " << hex << value << " is out of range"
                 << " (" << dec << ocpParam( P_mflag_wdth ) << " bits)" << "\n";
    }

    if ( numErrors > 0 )
        errors = errorStr.str();
    return ( numErrors > 0 );
}

//////////////////////////////////////////////////////////////////
// FUNCTION: StlParserSignalCommand::wrapup()
// DESCRIPTION: final processing. What is more practical to do now
// than during parsing.
// ARGUMENTS: None
// RETURNS: Nothing
//////////////////////////////////////////////////////////////////
void
StlParserSignalCommand::wrapup()
{
    // adjust the mask to the width
    if ( signal == S_MFlag ) {
        if (  value == static_cast<snx_uint64_t>(-1) )
            _value() &= static_cast<snx_uint64_t>
                ( ( 1ULL << ocpParam( P_mflag_wdth ) ) - 1 );
        if (  mask == static_cast<snx_uint64_t>(-1) )
            _mask() &= static_cast<snx_uint64_t>
                ( ( 1ULL << ocpParam( P_mflag_wdth ) ) - 1 );
    }        
}

//////////////////////////////////////////////////////////////////
// FUNCTION: StlParserWaitCommand::StlParserWaitCommand
// DESCRIPTION: Constructor:
// ARGUMENTS: ParamSet<OcpIp::ParamName>
//////////////////////////////////////////////////////////////////
StlParserWaitCommand::StlParserWaitCommand( const OcpParams& params ) :
    StlWaitCommand(),
    m_ocpParams( params )
{}

void
StlParserWaitCommand::clear()
{
    m_tmpWaiter.clear();
    _andor() = false; // "and" is default
    _waiters().clear();
}

int
StlParserWaitCommand::reportError( string& errors ) const
{
    const snx_uint64_t maxFlagVal = ( 1ULL << ocpParam( P_mflag_wdth ) ) -1;
    int threadWdth = OcpIp::ceilLog2( ocpParam( P_threads ) );
    const snx_uint64_t maxThreadBusyVal = ( 1ULL << threadWdth ) -1;
    uint32_t numErrors( 0 );
    ostringstream errorStr;

    deque<StlWaitCommand::Waiter>::const_iterator it;
    for ( it = waiters.begin(); it != waiters.end(); ++it ) {
        const Waiter& waiter = *it;
        if ( waiter.m_tbflags || waiter.m_response ) {
            // no way to verify this
            continue;
        }

        // check if the signal is defined
        bool exists = false;
        switch ( waiter.m_signal ) {
        case S_SError     : exists = ocpParam( P_serror )     ; break;
        case S_SInterrupt : exists = ocpParam( P_interrupt )  ; break;
        case S_ControlBusy: exists = ocpParam( P_controlbusy ); break;
        case S_StatusBusy : exists = ocpParam( P_statusbusy ) ; break;
        case S_ControlWr  : exists = ocpParam( P_controlwr )  ; break;
        case S_StatusRd   : exists = ocpParam( P_statusrd )   ; break;
        case S_SFlag      : exists = ocpParam( P_sflag )      ; break;
        case S_SThreadBusy: exists = ocpParam( P_sthreadbusy ); break;
        default: assert( false );
        }
        if ( !exists ) {
            numErrors++;
            errorStr << "wait signal " << s_ocp2Names.getName( waiter.m_signal )
                     << " is not enabled" << "\n";
        } else if ( waiter.m_signal == S_SFlag &&
                    waiter.m_value > maxFlagVal ) {
            numErrors++;
            errorStr << "wait value " << hex << waiter.m_value 
                     << " is out of range" << " ("
                     << dec << ocpParam( P_sflag_wdth ) << " bits)" << "\n";
        } else if ( waiter.m_signal == S_SThreadBusy &&
                    waiter.m_value > maxThreadBusyVal ) {
            numErrors++;
            errorStr << "wait value " << hex << waiter.m_value  
                     << " is out of range" << " ("
                     << dec << threadWdth << " bits)" << "\n";
        }
    }

    if ( numErrors > 0 )
        errors = errorStr.str();
    return ( numErrors > 0 );
}

//////////////////////////////////////////////////////////////////
// FUNCTION: StlParserControlStatusCommand::StlParserControlStatusCommand
// DESCRIPTION: Constructor:
// ARGUMENTS: ParamSet<OcpIp::ParamName>
//////////////////////////////////////////////////////////////////
void
StlParserControlStatusCommand::clear()
{
    _controlStatus() = false;
    _readWrite() = false;
    _value() = _mask() = 0;
}

int
StlParserControlStatusCommand::reportError( string& errors ) const
{ 
    bool isSystem = m_coreSystem;
    bool isCore   = !isSystem;
    
    uint32_t numErrors( 0 );
    ostringstream errorStr;
    // check for wrong system/core -ity
    if ( ( isSystem && ( ( isRead() && isControl() ) ||
                         ( isWrite() && isStatus() ) ) ) ||
         ( isCore && ( ( isRead() && isStatus() ) ||
                       ( isWrite() && isControl() ) ) ) ) {
        numErrors++;
        errorStr << ( isRead() ? "read" : "write" )
                 << ( isControl() ? "control" : "status" )
                 << " not allowed on "
                 << ( isCore ? "core" : "system" ) << " master\n";
    }
    bool exists = (
        ( isControl() && ocpParam( P_control ) ) ||
        ( isStatus() && ocpParam( P_status ) ) );
    if ( !exists ) {
        numErrors++;
        errorStr << ( isControl() ? "control" : "status" )
                 << " is not enabled\n";
    } else {
        uint32_t numBits = isControl() ?
            ocpParam( P_control_wdth ) : ocpParam( P_status_wdth );
        if ( value > ( ( 1ULL << numBits ) - 1 ) ) {
            numErrors++;
            errorStr << ( isControl() ? "control" : "status" )
                     << " is out of range (" << numBits << " bits)\n";
        }
    }

    if ( numErrors > 0 )
        errors = errorStr.str();
    return ( numErrors > 0 );
}

void
StlParserControlStatusCommand::wrapup()
{
    uint32_t numBits = isControl() ?
        ocpParam( P_control_wdth ) : ocpParam( P_status_wdth );

    // adjust the mask to the width
    if ( value == static_cast<snx_uint64_t>(-1) )
        _value() = ( 1ULL << numBits ) - 1;
    if ( mask == static_cast<snx_uint64_t>(-1) )
        _mask() = ( 1ULL << numBits ) - 1;
}
