// 
//  Copyright 2008 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG
//       Author : Anssi Haverinen, Nokia Inc.
//                Robert Guenzel (from TU of Braunschweig) for Greensocs Ltd.
//           $Id: slave.cc,v 1.1 2005/01/07 03:42:33 Anssi Exp $
//
//  Description : OCP API - TL1 AMBD profile example
// ============================================================================

#include "slave.h"

// ----------------------------------------------------------------------------
// Constructor
// ----------------------------------------------------------------------------
Slave::Slave (sc_core::sc_module_name name_)
  : sc_core::sc_module (name_), ipP("wPort"), req(NULL), req_data(NULL), ongiong_resp(false) {

  // initialize common members
  address = 0;
  bytemask = 0;
  for (int i=0;i<1024;i++)
    memory[i] = 0;

  SC_METHOD(proc);
  sensitive<<clk.pos();
  dont_initialize();
  ipP.register_nb_transport_fw(this, &Slave::nb_transport);
  ipP.activate_delta_cycle_protection();
  current_txn=NULL;
}


// ----------------------------------------------------------------------------
// Destructor
// ----------------------------------------------------------------------------
Slave::~Slave(){}


// ----------------------------------------------------------------------------
//  Method : Slave::proc()
//
//  Synchronous slave port process
//
// ----------------------------------------------------------------------------

void Slave::proc(){


  // Request handshake

  if (req) 
#ifdef DEBUG_G1
    std::cout << "Slave got request "
	 << " time " << sc_core::sc_time_stamp().to_seconds()
	 << " MCmd " << req->get_command()
	 << " MAddr " << req->get_address() << std::endl;
#endif
  
   
  if (req){
    if (pending_reqs.size()<8){//!req_fifo.full()) {
      // Burst addressing
      if (current_txn!=req){ //new req
        current_txn=req;
        chunk_cnt=0;
        address = req->get_address()/4; // First of burst
        ocpip::burst_length* b_len;
        ocpip::burst_sequence* b_seq;
        bool has_b_len=ipP.get_extension<ocpip::burst_length>(b_len, *req);
        bool has_b_seq=ipP.get_extension<ocpip::burst_sequence>(b_seq, *req);
        assert(has_b_len);
        unsigned int burstlen = b_len->value;
        if (has_b_seq)
          current_seq=b_seq->value.sequence;
        else
          current_seq=ocpip::INCR;
        if (ipP.get_extension<ocpip::srmd>(*req)){
          for (unsigned int i=0; i<burstlen; i++){
            pending_reqs.push_back(pending_req(req, address, i, i==burstlen-1));
            if (current_seq==ocpip::INCR)
              address = address+1; //incrementing address
            else 
            if (current_seq==ocpip::STRM)
              address = address; //streaming address              
          }
        }
        else
          pending_reqs.push_back(pending_req(req, address, 0, true));
      }
      else {
        chunk_cnt++;
        if (current_seq==ocpip::INCR)
          address = address+1; //incrementing address
        else 
        if (current_seq==ocpip::STRM)
          address = address; //streaming address
        pending_reqs.push_back(pending_req(req, address, chunk_cnt, true));
      }

      time=sc_core::SC_ZERO_TIME;
      //if (is_srmd) phase=ocpip::END_SRMD_REQ; else phase=tlm::END_REQ;
      phase=tlm::END_REQ;
      tlm::tlm_sync_enum retVal=ipP->nb_transport_bw(*req,phase, time);
      switch(retVal){
        case tlm::TLM_ACCEPTED: break;
        case tlm::TLM_UPDATED:
          std::cerr<<"Error in "<<name()<<" got unexpected phase update "<<phase<<" in response to END_(SRMD_)REQ"<<std::endl;
          exit(1);
          break;
        case tlm::TLM_COMPLETED:;
      }
#ifdef DEBUG_G1
      std::cout << "Slave accepted"<<((ipP.get_extension<ocpip::srmd>(*req))?" SRMD":"")<<" request "
           << " time " << sc_core::sc_time_stamp().to_seconds() << std::endl;
#endif
      req=NULL;
    }

  }
  // Data handshake
  if (pending_reqs.size() && (outstanding_rsps.size()<8)) {
    if (pending_reqs.front().txn->get_command()== tlm::TLM_WRITE_COMMAND){
      if (req_data) {
        assert(req_data==pending_reqs.front().txn);
        unsigned int MDataByteEn;
        byte_en_helper::setBE(MDataByteEn, req_data, sizeof(Td), pending_reqs.front().chunk);
        // Byte enable to bitmask
        if (MDataByteEn & 0x1)
          bytemask = 0xFF; 
        if  (MDataByteEn & 0x2)
          bytemask = 0xFF00 | bytemask; 
        if  (MDataByteEn & 0x4)
          bytemask = 0xFF0000 | bytemask; 
        if  (MDataByteEn & 0x8)
          bytemask = 0xFF000000 | bytemask; 
        
        memory[pending_reqs.front().address] = (*(((Td*)(req_data->get_data_ptr()))+pending_reqs.front().chunk)) & bytemask;
    #ifdef DEBUG_G1
        std::cout << "Slave got data "
             << " time " << sc_core::sc_time_stamp().to_seconds()
             << " MData " << (*(((Td*)(req_data->get_data_ptr()))+(pending_reqs.front().chunk))) << std::endl;
    #endif
        if (pending_reqs.front().resp) outstanding_rsps.push_back(outstanding_rsp(req_data, 0, pending_reqs.front().chunk));
        pending_reqs.pop_front();
        time=sc_core::SC_ZERO_TIME;
        phase=ocpip::END_DATA;
        tlm::tlm_sync_enum retVal=ipP->nb_transport_bw(*req_data,phase, time);
        switch(retVal){
          case tlm::TLM_ACCEPTED: break;
          case tlm::TLM_UPDATED:
            std::cerr<<"Error in "<<name()<<" got unexpected phase update "<<phase<<" in response to END_DATA"<<std::endl;
            exit(1);
            break;
          case tlm::TLM_COMPLETED: //completion should be impossible since everything end with a resp in this slave
            std::cerr<<"Error in "<<name()<<" got unexpected transaction completion "<<std::endl;
            exit(1);
        }
        req_data=NULL;
      }
    }
    else{
      outstanding_rsps.push_back(outstanding_rsp(pending_reqs.front().txn, pending_reqs.front().address, pending_reqs.front().chunk));
      pending_reqs.pop_front();
    }
  }

  // Response handshake
  if (outstanding_rsps.size()) {
    if (!ongiong_resp) {
      // Set OCP response
      outstanding_rsps.front().txn->set_response_status(tlm::TLM_OK_RESPONSE);
      if (outstanding_rsps.front().txn->get_command()==tlm::TLM_READ_COMMAND)
        *((Td*)(outstanding_rsps.front().txn->get_data_ptr())+(outstanding_rsps.front().chunk))=memory[outstanding_rsps.front().address];
      // Send response
      time=sc_core::SC_ZERO_TIME;
      phase=tlm::BEGIN_RESP;
      tlm::tlm_sync_enum retVal=ipP->nb_transport_bw(*outstanding_rsps.front().txn,phase, time);
      switch(retVal){
        case tlm::TLM_ACCEPTED: ongiong_resp=true; break;
        case tlm::TLM_UPDATED:
          if (phase!=tlm::END_RESP) {
            std::cerr<<"Error in "<<name()<<" got unexpected phase update "<<phase<<" in response to BEGIN_RESP"<<std::endl;
            exit(1);
          }
          break;
        case tlm::TLM_COMPLETED: break; //we don't check that it was really the last resp. we just trust the master...
      }


#ifdef DEBUG_G1
      std::cout << "Slave sent response "
	   << " time " << sc_core::sc_time_stamp().to_seconds()
	   << " SResp " << outstanding_rsps.front().txn->get_response_status() << std::endl;
#endif      
      outstanding_rsps.pop_front();
    }
  }
  
} // end of method
 
tlm::tlm_sync_enum Slave::nb_transport(tlm::tlm_generic_payload& txn, tlm::tlm_phase& ph, sc_core::sc_time& tim){
  switch(ph){
    case tlm::BEGIN_REQ: req=&txn;  break; //is_srmd=0;
    case tlm::END_RESP: ongiong_resp=false; break;
    default:
      //if (ph==ocpip::BEGIN_SRMD_REQ) {
      //  req=&txn; is_srmd=1;
      //}
      //else
      if (ph==ocpip::BEGIN_DATA){
        req_data=&txn;
      }
      else{
        std::cerr<<"Error in "<<name()<<" got unexpected phase "<<phase<<" in nb_transport_fw"<<std::endl;
        exit(1);        
      }
  }
  return tlm::TLM_ACCEPTED;
}
