// 
//  Copyright 2008 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG
//       Author : Anssi Haverinen, Nokia Inc.
//                Robert Guenzel (from TU of Braunschweig) for Greensocs Ltd.
//           $Id: slave.cc,v 1.2 2005/01/07 03:42:33 Anssi Exp $
//
//  Description : OCP API - TL1 profile example
// ============================================================================

#include "slave.h"

// ----------------------------------------------------------------------------
// Constructor
// ----------------------------------------------------------------------------
Slave::Slave (sc_core::sc_module_name name_)
  : sc_core::sc_module (name_), tpP("tpPort"), req(NULL), ongoing_rsp(false) {

  // initialize common members
  last_request = tlm::TLM_IGNORE_COMMAND;
  bytemask = 0;
  for (int i=0;i<1024;i++)
    memory[i] = 0;
  SC_METHOD(proc);
  sensitive<<clk.pos();
  dont_initialize(); // make sure we execute only at clock edge
  tpP.register_nb_transport_fw(this, &Slave::nb_transport);
  tpP.activate_synchronization_protection();
  rsp = NULL;
}


// ----------------------------------------------------------------------------
// Destructor
// ----------------------------------------------------------------------------
Slave::~Slave(){}


// ----------------------------------------------------------------------------
//  Method : Slave::proc()
//
//  Synchronous slave process
//
// ----------------------------------------------------------------------------

void Slave::proc(){

  // If a previous has been served, we check for new requests
  if (last_request == tlm::TLM_IGNORE_COMMAND) {
    if (req) {
      last_request = req->get_command();
      unsigned int MByteEn=0;
      byte_en_helper::setBE(MByteEn, req, sizeof(Td), 0);
      
      if (MByteEn & 0x1)
        bytemask = 0xFF; 
      if  (MByteEn & 0x2)
        bytemask = 0xFF00 | bytemask; 
      if  (MByteEn & 0x4)
        bytemask = 0xFF0000 | bytemask; 
      if  (MByteEn & 0x8)
        bytemask = 0xFF000000 | bytemask; 
      rsp=req;
#ifdef DEBUG_G1    
      if (last_request == tlm::TLM_WRITE_COMMAND){
        std::cout << "Slave got write request "
                  << "  time  = " << sc_core::sc_time_stamp().to_seconds()
                  << "  data  = " << (*((Td*)(rsp->get_data_ptr()))) << std::endl;
      }
      else{
        std::cout << "Slave got read request "
                  << "  time  = " << sc_core::sc_time_stamp().to_seconds()
                  << "  address  = " << rsp->get_address() << std::endl;
      }
#endif
      time=sc_core::SC_ZERO_TIME;
      phase=tlm::END_REQ;
      tlm::tlm_sync_enum retVal=tpP->nb_transport_bw(*req,phase, time);
      switch(retVal){
        case tlm::TLM_ACCEPTED: break;
        case tlm::TLM_UPDATED:
          std::cerr<<"Error in "<<name()<<" got unexpected phase update "<<phase<<" in response to END_REQ"<<std::endl;
          exit(1);
          break;
        case tlm::TLM_COMPLETED:;
      }
      req=NULL;
    }
  }


  if (last_request == tlm::TLM_WRITE_COMMAND){
    assert(rsp);
    assert(tpP.get_extension<ocpip::posted>(*rsp)); //we do not support nonposted writes
    memory[rsp->get_address()/4] = (*((Td*)(rsp->get_data_ptr()))) & bytemask;
  }

  if (!ongoing_rsp && rsp) {
    if (last_request == tlm::TLM_READ_COMMAND)
      *((Td*)(rsp->get_data_ptr())) = memory[rsp->get_address()/4];
    // Set OCP response
    rsp->set_response_status(tlm::TLM_OK_RESPONSE);
    
    // Send response
    time=sc_core::SC_ZERO_TIME;
    phase=tlm::BEGIN_RESP;
    tlm::tlm_sync_enum retVal=tpP->nb_transport_bw(*rsp,phase, time);
    switch(retVal){
      case tlm::TLM_ACCEPTED: ongoing_rsp=true; break;
      case tlm::TLM_UPDATED:
        if (phase!=tlm::END_RESP) {
          std::cerr<<"Error in "<<name()<<" got unexpected phase update "<<phase<<" in response to BEGIN_RESP"<<std::endl;
          exit(1);
        }
        break;
      case tlm::TLM_COMPLETED: break;
    }

#ifdef DEBUG_G1
    std::cout << "Slave sent response " << rsp->get_response_status()
              << " time " << sc_core::sc_time_stamp().to_seconds();
    if (last_request == tlm::TLM_READ_COMMAND)
      std::cout<< " data " << (*((Td*)(rsp->get_data_ptr())));
    std::cout<< std::endl;
#endif
    
    last_request = tlm::TLM_IGNORE_COMMAND;
    rsp=NULL;
  }
} // end of method

tlm::tlm_sync_enum Slave::nb_transport(tlm::tlm_generic_payload& txn, tlm::tlm_phase& ph, sc_core::sc_time& tim){
  switch(ph){
    case tlm::BEGIN_REQ: req=&txn; break;
    case tlm::END_RESP: ongoing_rsp=false; break;
    default:
      std::cerr<<"Error in "<<name()<<" got unexpected phase "<<phase<<" in nb_transport_fw"<<std::endl;
      exit(1);        
  }
  return tlm::TLM_ACCEPTED;
}


