// Copyright (c) 2004 Sonics, Inc.
//
// Confidential and Proprietary Information of Sonics, Inc.
// Use, disclosure, or reproduction is prohibited without
// written permission from Sonics, Inc.
//
// $Id: StlParserAction.h,v 1.2 2007/10/08 23:46:44 halexan Exp $

#ifndef _OcpIp_StlParserAction_h_
#define _OcpIp_StlParserAction_h_

#include <functional>
#include "BigInt.h"

template <typename numT, typename iteratorT>
numT
convertNum( iteratorT first, iteratorT last )
{
    char base = 10;
    if ( last - first > 2 && *first == '0' && *(first+1) == 'x' )
        base = 16;
    return static_cast<numT>( strtoull( first, 0, base ) );
}

template <typename numT>
struct SetNumProxy {
    // This class is a proxy to a numeric value.
    // either we specify the value to assign at construction time in which case
    // the numeric action (operator( numT ) ) should not be called
    // or we assign the value through the numeric action
    SetNumProxy( numT& num ) :
        m_num( num ), m_val( num ), m_fixed( false ) {}
    SetNumProxy( numT& num, numT val ) : 
        m_num( num ), m_val( val ), m_fixed( true ) {}
    void operator()( numT val ) const {
        m_num = m_fixed ? m_val : val;
    }
    template <typename iteratorT>
    void operator()( iteratorT first, iteratorT last ) const {
        operator()( convertNum<numT>( first, last ) );
    }
 
    numT&       m_num;
 private:
    numT        m_val;
    bool        m_fixed;
};

template <typename numT>
struct SetPairProxy {
    // This class is a proxy to a pair of values represented in one rule
    SetPairProxy( numT& num1, numT& num2, const string& separators=" " ) :
        m_numP1( num1 ), m_numP2( num2 ), m_separators( separators ) {}
    template <typename iteratorT>
    void operator()( iteratorT first, iteratorT last ) const {
        if ( *(last-1) == ')' ) last = last-1;
        string str( first, last );
        size_t firstSep = str.find_first_of( m_separators );
        if ( firstSep != string::npos ) {
            m_numP1( first, first + firstSep - 1);
            m_numP2( first + firstSep + 1, last );
        } else {
            m_numP1( first, last );
        }
    }
  private:
    SetNumProxy< numT > m_numP1, m_numP2;
    const string m_separators;
};

template <typename numT>
SetNumProxy<numT> setVal( numT& num ) {
    return SetNumProxy<numT>( num );
}
template <typename numT>
SetNumProxy<numT> setVal( numT& num, numT val ) {
    return SetNumProxy<numT>( num, val );
}
template <typename numT>
SetPairProxy<numT> setPair( numT& num1, numT& num2, const string& separators ) {
    return SetPairProxy<numT>( num1, num2, separators );
}

struct SetVersionProxy {
    // This class is a proxy to a pair of values represented in one rule
    SetVersionProxy( bool& versionOk ) :
        m_versionOk( versionOk ) {}
    template <typename iteratorT>
    void operator()( iteratorT first, iteratorT last ) const {
        m_versionOk = ( string( first, last ) == "2.0" );
    }
  private:
    bool& m_versionOk;
};

struct SetCmdProxy {
    SetCmdProxy( ocpip_legacy::OcpIp::StlParser::MyTransferCommand& transfer ) :
        transfer( transfer ) {}
    void operator()( const char* first, const char* last ) const {
	using namespace ocpip_legacy;

        string cmd( first, last );
        int (*pf)(int) = tolower;
        transform( cmd.begin(), cmd.end(), cmd.begin(), pf );
        size_t wdthIt = cmd.find_first_of( "0123456789" );
        if ( wdthIt != string::npos )
            transfer.m_transferWidth = convertNum<uint32_t>(
                first + wdthIt, last );
        assert( !cmd.empty() );

        // strip the leading b, it won't influence the command
        if ( cmd[0] == 'b' && cmd != "bcst" && cmd != "broadcast" )
            cmd = cmd.substr( 1, wdthIt );
        else
            cmd = cmd.substr( 0, wdthIt );
        
        OCPMCmdType& mCmd = transfer._request().MCmd;
        if ( cmd == "write" || cmd == "wr" )
            mCmd = OCP_MCMD_WR;
        else if ( cmd == "writenonpost" || cmd == "wrnp" )
            mCmd = OCP_MCMD_WRNP;
        else if ( cmd == "writeconditional" || cmd == "wrc" )
            mCmd = OCP_MCMD_WRC;
        else if ( cmd == "broadcast" || cmd == "bcst" )
            mCmd = OCP_MCMD_BCST;
        else if ( cmd == "read" || cmd == "rd" )
            mCmd = OCP_MCMD_RD;
        else if ( cmd == "readex" || cmd == "rdex" )
            mCmd = OCP_MCMD_RDEX;
        else if ( cmd == "readlinked" || cmd == "rdl" )
            mCmd = OCP_MCMD_RDL;
        else
            assert( false );
    }
 private:
    ocpip_legacy::OcpIp::StlParser::MyTransferCommand& transfer;
};

struct BurstMacroProxy {
    BurstMacroProxy( ocpip_legacy::OcpIp::StlParser::MyTransferCommand& transfer ) :
        transfer( transfer ) {}
    void operator()( const char*, const char* ) const {
	ocpip_legacy::OcpIp::StlParser::MyTransferCommand::Request& request = transfer._request();
        request.DataLength = request.MBurstLength;
        if ( request.MBlockHeight > 1 )
	    request.DataLength *= request.MBlockHeight;
    }
 private:
    ocpip_legacy::OcpIp::StlParser::MyTransferCommand& transfer;
};

#endif /* _OcpIp_StlParserAction_h_ */
