///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// (c) Copyright OCP-IP 2008
// OCP-IP Confidential and Proprietary
//
//
//============================================================================
//      Project : OCP SLD WG
//       Author : James Aldis, Texas Instruments
//                Robert Guenzel (from TU of Braunschweig) for Greensocs Ltd.
//
//          $Id:
//
//  Description :  The TL1 timing example. It is equivalent to the example with
//                 the same name in the OCP TL1 channel kit, and it produces
//                 100% the same results.
//                 It tests the correct functionality of timing information
//                 distribution and (non pipelined) thread busy exact control
//                 flow.
//
//                                                                           //
///////////////////////////////////////////////////////////////////////////////


#include <string>
#include <map>
#include "deep_copier.h"
#include "timing_master.h"
#include "timing_slave.h"
#include "timing_merger.h"
#include "timing_splitter.h"


// programme
int sc_main(int argc, char **argv)
{
  // simulator configuration
  sc_core::sc_set_time_resolution(1.0,sc_core::SC_PS);

  double duration = 5000.0;
  double clk_period = 5.0;

  // **** SIMULATION CONFIGURATION ****
  std::cout << "usage: ./tl1_timing_test [d duration] [c clk-period]" << std::endl;
  std::cout << "  using defaults for unspecified parameters" << std::endl;

  char **nextarg = argv;
  int argnr = 1;
  while((argnr+1) < argc) {
    nextarg++;
    argnr += 2; // always have a letter then a parameter
    switch(**nextarg) {
      case 'd':
        sscanf(*(++nextarg),"%lf",&duration);
        break;
      case 'c':
        sscanf(*(++nextarg),"%lf",&clk_period);
        break;
      default:
        std::cout << "unknown command-line parameter - " << **nextarg << " - trying to continue\n";
        argnr--;
    }
  }

  if(clk_period < 1.5) {
    std::cout << "Clock period specified is too big\n";
    std::cout << "  => setting it to 2 ns\n";
    clk_period = 2.0;
  }

  unit_rand("init");
  get_address_random("init");

  // **** MODULE INSTANTIATION ****
  // clock source
  sc_core::sc_time clk_per(clk_period, sc_core::SC_NS);
  sc_core::sc_time dur(duration, sc_core::SC_NS);
  sc_core::sc_clock clk("clk",clk_per, 0.5);

  // a collection of tl1_timing_master, tl1_timing_slave,
  // tl1_timing_splitter and tl1_timing_merger, making a (fully
  // combinatorial) network, with OCP-TL1-channels between them.
  // in this case we have a 4x4 shared bus.

  // masters are single-threaded; one of them can change REQ delay
  // on the fly
  timing_master mA("mA", false);
  timing_master mB("mB", false);
  timing_master mC("mC", true);
  timing_master mD("mD", false);

  // mergers have (N,M) threads on slave ports and N+M on master port
  timing_merger mgAB("mgAB", 1, 1);
  timing_merger mgCD("mgCD", 1, 1);
  timing_merger mgABCD("mgABCD", 2, 2);

  // splitters have N threads on slave port and (N,N) on master ports
  timing_splitter spABCD("spABCD", 4);
  timing_splitter spAB("spAB", 4);
  timing_splitter spCD("spCD", 4);

  // slaves are N-threaded
  timing_slave sA("sA", 4);
  timing_slave sB("sB", 4);
  timing_slave sC("sC", 4);
  timing_slave sD("sD", 4);

  // **** OTHER BINDINGS ****
  mA.clk(clk);
  mB.clk(clk);
  mC.clk(clk);
  mD.clk(clk);
  sA.clk(clk);
  sB.clk(clk);
  sC.clk(clk);
  sD.clk(clk);
  mgAB.clk(clk);
  mgCD.clk(clk);
  mgABCD.clk(clk);
  spABCD.clk(clk);
  spAB.clk(clk);
  spCD.clk(clk);

#ifdef WITH_DEEP_COPY
  deep_copier<32> dp_cpy("Copier");
#endif

#ifdef USE_OCP_MONITOR
  ocpip::ocp_connection_monitor<32> ocp_monitor_t_piece1(mA.ocp, mgAB.ocpsA);
  ocpip::ocp_tl1_monitor_adapter<32,32> mon_adapt1(ocp_monitor_t_piece1);

  ocpip::ocp_connection_monitor<32> ocp_monitor_t_piece2(mgABCD.ocpm,spABCD.ocps);
  ocpip::ocp_tl1_monitor_adapter<32,32> mon_adapt2(ocp_monitor_t_piece2);

  ocpip_legacy::OCP_TL1_Trace_Monitor_Clocked<ocpip::ocp_data_class_unsigned<32,32> > tracer1("Tracer1", "mAocp.ocp");
  tracer1.p_mon(mon_adapt1);
  tracer1.p_clk(clk);

  ocpip_legacy::OCP_TL1_Trace_Monitor_Clocked<ocpip::ocp_data_class_unsigned<32,32> > tracer2("Tracer2", "mgABCDocp.ocp");
  tracer2.p_mon(mon_adapt2);
  tracer2.p_clk(clk);
#else
  mA.ocp(mgAB.ocpsA);
  mgABCD.ocpm(spABCD.ocps);
#endif

  mB.ocp(mgAB.ocpsB);
  mC.ocp(mgCD.ocpsA);
  mD.ocp(mgCD.ocpsB);
  mgAB.ocpm(
#ifdef WITH_DEEP_COPY
    dp_cpy.ocps);
    dp_cpy.ocpm(
#endif
  mgABCD.ocpsA);
  mgCD.ocpm(mgABCD.ocpsB);
  spABCD.ocpmA(spAB.ocps);
  spABCD.ocpmB(spCD.ocps);
  spAB.ocpmA(sA.ocp);
  spAB.ocpmB(sB.ocp);
  spCD.ocpmA(sC.ocp);
  spCD.ocpmB(sD.ocp);

  // **** SIMULATION ****
  std::cout << "Starting simulation" << std::endl;
  std::cout << "  duration = " << duration << std::endl;
  std::cout << "  clk frequency = " << 1.0/clk_period <<std::endl;

  sc_core::sc_start(dur);
  sc_core::sc_stop();
  std::cout << "Simulation complete " << std::endl;

  unit_rand("finalize");
  get_address_random("finalize");

  std::cout<<sc_core::sc_delta_count()<<std::endl;

  return(0);
}

