//////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2005 Sonics, Inc.
//
// Confidential and Proprietary Information of Sonics, Inc.
// Use, disclosure, or reproduction is prohibited without
// written permission from Sonics, Inc.
//
// $Id: MemoryCl.h,v 1.1 2007/01/25 21:51:22 halexan Exp $
//
//////////////////////////////////////////////////////////////////////
#ifndef _OcpipSlaveMemoryCl_h
#define _OcpipSlaveMemoryCl_h

#include <ostream>
#include <map>
#include <string>
#include <bitset>
#include <math.h>
#include "ocp_utils.h"

namespace OcpIp {

typedef enum {IM_UNKNOWN = 0,
	      IM_FIXED,
	      IM_RANDOM,
	      IM_ADDRESS,
              IM_REQINFO  // For debug: return reqinfo, discard writes (not used here)
} InitMode;

// define the MemoryCl class
template <class Td, class Ta>
class MemoryCl {

  private:

    template<typename numT>
    static numT fillRand(unsigned int numbits = 8*sizeof( numT ) )
    {
        numT          rslt = 0; // default constructor may create Xs
        unsigned int  nb   = numbits;
        unsigned int  shft = 8 * sizeof(int);

        for (; nb > 0; nb -= shft) { // do int at a time
            numT rnd((unsigned int)random()); // avoid sign extend
	    if (shft > nb)     {
	        // may need to feed in an X here 
                rnd  >>= (shft - nb);
                shft   = nb;
            }
	    if (nb != numbits) {
                rslt <<= shft;
            }
  	    rslt |= rnd;
        }

        return rslt;
    }

    void showMasks() {
        cout << "m_LastByteEn "      << hex << m_LastByteEn      << endl;
        cout << "m_LastNewDataMask " << hex << m_LastNewDataMask << endl;
        cout << "m_LastOldDataMask " << hex << m_LastOldDataMask << endl;
    }

    inline void fixMasks(unsigned int byteen)
    {
        if (byteen != m_LastByteEn) {
            // handle the special (which is very common)
            if (byteen == m_ByteEnMask) {
                m_LastOldDataMask = 0;
                m_LastNewDataMask = ~m_LastOldDataMask;
            } else {
                recalculateByteMasks(byteen);
            }

            // save the byteen value
            m_LastByteEn = byteen;
        }
        // showMasks();
    }

  public:
    // --------------------------
    // public members and methods
    // --------------------------

    // type definitions
    typedef map< Ta, Td > MemMapType;

    // constructor
    // - address width in number of bits
    // - word size in number of bytes
    MemoryCl(string name, unsigned int addr_width, unsigned int word_size,
             InitMode mode = IM_UNKNOWN, Td meminit_fixedbytedata = 0)
      : m_Name(name),
        m_WordSize(word_size),
        m_InitMode(mode),
        m_FixedData(0),
        m_AddressOffset(0),
        m_LastByteEn(0)     // forces mask calculation 
    {
        unsigned int  i;
        Td            tmp_data_value;

        checkNumericTypes();

        // setup content of the number of shift bits
        m_AddrShiftRight = OcpIp::ceilLog2(word_size);

        // setup the byte enable mask
        setupByteEnMasks(word_size);

        // ---------------------------------
        // setup content of the address mask
        // ---------------------------------

        // adjust the address width, if needed
        if (addr_width > (sizeof(Ta)*8)) {
            cout << "Address Width (" << addr_width
                 << ") cannot be greater than "
                 << (sizeof(Ta)*8) << endl;
            addr_width = sizeof(Ta)*8;
        }

        // calculate the address mask
        OcpIp::fillBitRange( m_AddrMask, addr_width-1 );

        // calculate the per word fixed data value for uninitialized
        //   memory location, if needed
        if (m_InitMode == IM_FIXED) {
            // get the per byte fixed data value
            tmp_data_value = (Td) (meminit_fixedbytedata & 0xFF);

            // set the per word fixed data value
            for (i = 0; i < m_WordSize; i++) {
                m_FixedData = m_FixedData | tmp_data_value;
                tmp_data_value = tmp_data_value << 8;
            }
            //cout << "m_FixedData = " << hex << m_FixedData << endl;
        }
    }

    // destructor
    virtual ~MemoryCl(void)
    { }

    void
    clear(void)
    {
        // erases the memory
        m_Memory.clear();
        fixMasks(0);
    }
        
    void
    dumpState(ostream& os = std::cout)
    {
        typename MemMapType::iterator it;
        unsigned int         i = 0;

        os << hex << "Memory " << m_Name << endl;
        for (it = m_Memory.begin(); it != m_Memory.end(); ++it) {
            os << "Memory[" << it->first << "] = 0x" << hex
               << it->second << endl;
            i++;
        }
        os << "Memory size = " << i << endl;
    }

    // methods
    void
    read( const Ta& addr, Td& data, unsigned int byteen)
    {
        Td data_value; // defaulted to get X if 4-state

        //
        byteen &= m_ByteEnMask;
        if (byteen == 0) {
            // return all-zeros data
            data = 0;
            return;
        }

        fixMasks(byteen);

        Ta my_addr = (addr >> m_AddrShiftRight);

        // check whether the memory location has be initialized
        if (m_Memory.count(my_addr) == 0) {
            // read an uninitialized memory location
            switch (m_InitMode) {
	    case IM_FIXED:   data_value = m_FixedData;            break;
            case IM_ADDRESS: data_value = addr + m_AddressOffset; break;
	    case IM_RANDOM:  data_value = fillRand<Td>();
                             // Can't regenerate so need to save
                             m_Memory.insert( make_pair( my_addr, Td( data_value ) ) );
                             break;
	    case IM_UNKNOWN: // Assume default constructor for data_value does this
                             break;
            default:         // return all-zero data 
                             data_value = 0;
	                     return;
	    }
        } else {
            // get the data value first
            data_value = m_Memory[my_addr];
        }

        //byte_en:
        // only return the byte enabled part
        data = (m_LastNewDataMask & data_value) |
               (m_LastOldDataMask & data);
        return;
    }

    void
    write(const Ta& addr, const Td& data, unsigned int byteen)
    {
        Td data_value;

        //
        byteen &= m_ByteEnMask;
        if (byteen == 0) return;

        Ta my_addr = (addr >> m_AddrShiftRight);

        // check whether the memory location has be initialized
        if (m_Memory.count(my_addr) == 0) {
	    // call read to initialize properly,
            //           use same byteen to avoid doing mask work twice 
	    read(addr,data_value,byteen); 
        } else {
            data_value = m_Memory[my_addr];
            fixMasks(byteen);
	}

        // only update the byte enabled part        
	m_Memory.insert( make_pair( my_addr, Td() ) );
        m_Memory[my_addr] = 
            (m_LastNewDataMask & data) | (m_LastOldDataMask & data_value);

        //
        //cout << "Memory[" << my_addr <<"] " << m_Memory[my_addr] << endl;
        //cout << "m_LastNewDataMask " << m_LastNewDataMask << endl;
        //cout << "m_LastOldDataMask " << m_LastOldDataMask << endl;
        //cout << "addr " << addr << ", data " << data << ", byteen "
        //     << byteen << endl;

        //dumpState();
    }

  private:

    static bool checkNumericTypes()
    {
        return sizeof( Ta ) <= 8;
    }

    //
    void
    recalculateByteMasks(unsigned int byteen)
    {
        // re-calculate the masks
        bitset<16>    my_byte_en = byteen;
        unsigned int  i;
        Td            curByteMask(0xFF);
        Td            newMask(0);

        // rebuild the masks
        for (i = 0; i < m_NumByteEnBits; i++) {
               // update the current byte mask and negative mask
            if (my_byte_en[i]) newMask |= curByteMask;

            // reset the current byteen
            my_byte_en[i] = 0;
            if (my_byte_en == 0) break;            

                // update the masks, if the current byte is enabled
	        curByteMask <<= 8;
        }

        m_LastNewDataMask =  newMask;
        m_LastOldDataMask = ~newMask;
    }

    void
    setupByteEnMasks(unsigned int word_size)
    {        
        m_NumByteEnBits = word_size;
        m_ByteEnMask = ( 1U << word_size ) - 1;
        if ( word_size >= 32 ) {
            cout << "Non-supported Word Size of "
                 << word_size << " (" << word_size * 8 << " bits) found." << endl;
            exit(-1);
        }
        fixMasks(m_ByteEnMask);
    }

  public:
    const string       m_Name;
    const unsigned int m_WordSize;  // in bytes
    const InitMode     m_InitMode;

  private:
    //
    unsigned int m_AddrShiftRight;
    unsigned int m_ByteEnMask;
    unsigned int m_NumByteEnBits;

    Td           m_FixedData;
    Ta           m_AddrMask;
    Ta           m_AddressOffset;
    Td           m_OldDataMask;

    //
    unsigned int m_LastByteEn;
    Td           m_LastOldDataMask;
    Td           m_LastNewDataMask;
    MemMapType   m_Memory;

};
}

#endif // _OcpipSlaveMemoryCl_h
