class i2c_base_seq extends uvm_sequence #(i2c_packet); 
  `uvm_object_utils(i2c_base_seq)

  function new(string name = "i2c_base_seq");
    super.new(name);
    //set_automatic_phase_objection(1);
  endfunction 

  virtual task pre_body();
  uvm_phase phase;
  `ifdef UVM_VERSION_1_2
    // in UVM1.2, get starting phase from method
    phase = get_starting_phase();
  `else
    phase = starting_phase;
  `endif
  if (phase != null) begin
    phase.raise_objection(this, get_type_name());
    `uvm_info(get_type_name(), "raise objection", UVM_MEDIUM)
  end
  endtask : pre_body

  virtual task post_body();
  uvm_phase phase;
  `ifdef UVM_VERSION_1_2
    // in UVM1.2, get starting phase from method
    phase = get_starting_phase();
  `else
    phase = starting_phase;
  `endif
  if (phase != null) begin
    phase.drop_objection(this, get_type_name());
    `uvm_info(get_type_name(), "drop objection", UVM_MEDIUM)
  end
  endtask : post_body

  virtual task body(); 
    `uvm_info(get_type_name(), "Call i2c_base_seq sequence", UVM_LOW)
  endtask

endclass : i2c_base_seq

// Valid sequences : 
class r_seq extends i2c_base_seq; // Read from target
  `uvm_object_utils(r_seq)

  function new(string name="r_seq");
    super.new(name);
  endfunction

  virtual task body(); 
    `uvm_do_with(req, {rw_ == 1; tg_addr_valid == VALID_TG_ADDR; reg_addr_valid == VALID_REG_ADDR; })
  endtask
endclass : r_seq

class w_seq extends i2c_base_seq; // Write to target
  `uvm_object_utils(w_seq)

  function new(string name="w_seq");
    super.new(name);
  endfunction

  virtual task body(); 
    `uvm_do_with(req, {rw_ == 0; tg_addr_valid == VALID_TG_ADDR; reg_addr_valid == VALID_REG_ADDR; })
  endtask
endclass : w_seq

// Faulty sequences : 

class r_invalid_target_id_seq extends i2c_base_seq; // Try to read with an invalid target id
  `uvm_object_utils(r_invalid_target_id_seq)

  function new(string name="r_invalid_target_id_seq");
    super.new(name);
  endfunction

  virtual task body(); 
    `uvm_do_with(req, {rw_ == 1; tg_addr_valid == INVALID_TG_ADDR; reg_addr_valid == VALID_REG_ADDR; })
  endtask
endclass : r_invalid_target_id_seq

class w_invalid_target_id_seq extends i2c_base_seq; // Try to write with invalid target id
  `uvm_object_utils(w_invalid_target_id_seq)

  function new(string name="w_invalid_target_id_seq");
    super.new(name);
  endfunction

  virtual task body(); 
    `uvm_do_with(req, {rw_ == 0; tg_addr_valid == INVALID_TG_ADDR; reg_addr_valid == VALID_REG_ADDR; })
  endtask
endclass : w_invalid_target_id_seq

class r_invalid_reg_id_seq extends i2c_base_seq; // Try to read with invalid register address
  `uvm_object_utils(r_invalid_reg_id_seq)

  function new(string name="r_invalid_reg_id_seq");
    super.new(name);
  endfunction

  virtual task body(); 
    `uvm_do_with(req, {rw_ == 1; tg_addr_valid == VALID_TG_ADDR; reg_addr_valid == INVALID_REG_ADDR; })
  endtask
endclass : r_invalid_reg_id_seq

class w_invalid_reg_id_seq extends i2c_base_seq; // Try to write with invalid register address
  `uvm_object_utils(w_invalid_reg_id_seq)

  function new(string name="w_invalid_reg_id_seq");
    super.new(name);
  endfunction

  virtual task body(); 
    `uvm_do_with(req, {rw_ == 0; tg_addr_valid == VALID_TG_ADDR; reg_addr_valid == INVALID_REG_ADDR; })
  endtask
endclass : w_invalid_reg_id_seq

class i2c_invalid_id_read_seq extends i2c_base_seq; // valid write followed by a read with an invalid target address followed by a valid read
  `uvm_object_utils(i2c_invalid_id_read_seq)

  function new(string name="i2c_invalid_id_read_seq"); 
    super.new(name);
  endfunction
  
  rand bit [7:0] random_reg_addr;
  constraint c1 {random_reg_addr inside {[0:2]};} 

  virtual task body(); 
    `uvm_info(get_type_name(), "Call i2c_invalid_id_read_seq", UVM_LOW)
    `uvm_do_with(req, {rw_ == 0; tg_addr_valid == VALID_TG_ADDR; reg_addr == random_reg_addr;}) // write
    `uvm_do_with(req, {rw_ == 1; tg_addr_valid == INVALID_TG_ADDR; reg_addr_valid == VALID_REG_ADDR;}) // read with invalid target address to make sure repeated start gets discarded
    `uvm_do_with(req, {rw_ == 1; tg_addr_valid == VALID_TG_ADDR; reg_addr == random_reg_addr;}) // read with valid target address to test new start condition is detected
  endtask
endclass : i2c_invalid_id_read_seq

class i2c_invalid_reg_addr_seq extends i2c_base_seq; // valid write followed by a read with an invalid target address 
  `uvm_object_utils(i2c_invalid_reg_addr_seq)

  function new(string name="i2c_invalid_reg_addr_seq"); 
    super.new(name);
  endfunction
  
  bit [7:0] reg_addr_param = 1;
  virtual task body(); 
    `uvm_info(get_type_name(), "Call i2c_invalid_reg_addr_seq", UVM_LOW)
    `uvm_do_with(req, {rw_ == 0; tg_addr_valid == VALID_TG_ADDR; reg_addr_valid == VALID_REG_ADDR; reg_addr == reg_addr_param;}) // write
    `uvm_do_with(req, {rw_ == 1; tg_addr_valid == VALID_TG_ADDR; reg_addr_valid == INVALID_REG_ADDR;}) // read with invalid register address to make sure repeated start gets discarded
    `uvm_do_with(req, {rw_ == 1; tg_addr_valid == VALID_TG_ADDR; reg_addr_valid == VALID_REG_ADDR; reg_addr == reg_addr_param;}) // read with valid register address to test new start condition is detected
  endtask
endclass : i2c_invalid_reg_addr_seq

class i2c_wr_seq extends i2c_base_seq; 
  `uvm_object_utils(i2c_wr_seq)

  function new(string name="i2c_wr_seq"); 
    super.new(name);
  endfunction

  rand bit [7:0] rg_addr;

  constraint rg_c {rg_addr inside {[0:2]};}

  virtual task body(); 
    `uvm_info(get_type_name(), "Call i2c_wr_seq sequence", UVM_LOW)
    
    `uvm_do_with(req, {rw_ == 0; tg_addr_valid == VALID_TG_ADDR; reg_addr == rg_addr;})
    `uvm_do_with(req, {rw_ == 1; tg_addr_valid == VALID_TG_ADDR; reg_addr == rg_addr;}) 
  endtask

endclass : i2c_wr_seq

class i2c_wrw_seq extends i2c_base_seq; 
  `uvm_object_utils(i2c_wrw_seq)

  function new(string name="i2c_wrw_seq"); 
    super.new(name);
  endfunction

  rand bit [7:0] rg_addr;
  rand bit [7:0] tg_addr;

  constraint rg_c {rg_addr inside {[0:2]};}
  constraint tg_c {tg_addr == 8'h00;}

  virtual task body(); 
    `uvm_info(get_type_name(), "Call i2c_wrw_seq sequence", UVM_LOW)
    
    `uvm_do_with(req, {rw_ == 0; tg_addr_valid == VALID_TG_ADDR; reg_addr == rg_addr;})
    `uvm_do_with(req, {rw_ == 1; tg_addr_valid == VALID_TG_ADDR; reg_addr == rg_addr;}) 
    `uvm_do_with(req, {rw_ == 0; tg_addr_valid == VALID_TG_ADDR; reg_addr_valid == VALID_REG_ADDR;})
  endtask

endclass : i2c_wrw_seq

class i2c_init_seq extends i2c_base_seq; 
  `uvm_object_utils(i2c_init_seq)

  function new(string name="i2c_init_seq"); 
    super.new(name);
  endfunction

  virtual task body(); 
    `uvm_info(get_type_name(), "Call i2c_init_seq", UVM_LOW)
    // writing to addr 1-2-3
    `uvm_do_with(req, {rw_ == 0; tg_addr_valid == VALID_TG_ADDR; reg_addr == 0; data == 8'b0000_0000;})
    `uvm_do_with(req, {rw_ == 0; tg_addr_valid == VALID_TG_ADDR; reg_addr == 1; data == 8'b0000_0100;}) 
    `uvm_do_with(req, {rw_ == 0; tg_addr_valid == VALID_TG_ADDR; reg_addr == 2; data == 8'b0000_0000;})
  endtask

endclass : i2c_init_seq

class i2c_rdm_mux_seq extends i2c_base_seq; 
  `uvm_object_utils(i2c_rdm_mux_seq)

  function new(string name="i2c_rdm_mux_seq"); 
    super.new(name);
  endfunction

  i2c_init_seq init;

  virtual task body(); 
    `uvm_info(get_type_name(), "Call i2c_rdm_mux_seq", UVM_LOW)
    // initialize
    `uvm_do(init);
    repeat(5) // fully random programming of registers
      `uvm_do_with(req, {rw_ == 0; tg_addr_valid == VALID_TG_ADDR; reg_addr_valid == VALID_REG_ADDR;})
    repeat(5) // targeted random programming of freq mux selection
      `uvm_do_with(req, {rw_ == 0; tg_addr_valid == VALID_TG_ADDR; reg_addr == 1; data inside {[8'h04:8'h07]};})
	// reset mux to div4
    `uvm_do_with(req, {rw_ == 0; tg_addr_valid == VALID_TG_ADDR; reg_addr == 1; data == 8'b0000_0100;})
  endtask

endclass : i2c_rdm_mux_seq
