// tb class for digital version of frequency adapter
class freq_adpt_tb extends uvm_env;

  // component macro
  `uvm_component_utils(freq_adpt_tb)

  //registers_env registers;
  i2c_env i2c;
  osc_env freq_generator;
  osc_env freq_detector; 
  
  freq_adpt_scoreboard freq_adpt_sb;

  // Constructor
  function new (string name, uvm_component parent=null);
    super.new(name, parent);
  endfunction : new

  // UVM build() phase
  function void build_phase(uvm_phase phase);
    `uvm_info("MSG","In the build phase",UVM_MEDIUM)
    
    // set up virtual interfaces for UVCs and scoreboard
    uvm_config_db#(virtual osc_if)::set(this,"freq_generator*","vif", top.generator_if);
    uvm_config_db#(virtual osc_if)::set(this,"freq_detector*", "vif", top.detector_if);
    uvm_config_db#(virtual i2c_if)::set(this,"i2c.agent.*", "vif", top.i2c_if);

    // config the value of diff_sel for freq_generator to 0 - single-ended clock generation
    uvm_config_int::set(this,"freq_generator.agent.*","diff_sel", 0); 
    // config the value of diff_sel for freq_detector to 1 - differential clock detection
    uvm_config_int::set(this,"freq_detector.agent.*","diff_sel", 1);
    
    super.build_phase(phase);
    
    // create the envs for the generator, detector, registers and scoreboard
    freq_generator  = osc_env::type_id::create("freq_generator",  this);
    freq_detector   = osc_env::type_id::create("freq_detector",   this);
    i2c             = i2c_env::type_id::create("i2c", this);
    freq_adpt_sb    = freq_adpt_scoreboard::type_id::create("freq_adpt_sb",   this);

  endfunction : build_phase

  // UVM connect_phase
  function void connect_phase(uvm_phase phase);
    // Connect the TLM ports from the UVCs to the scoreboard
    //registers.reg_agent.monitor.item_collected_port.connect(freq_adpt_sb.sb_registers_in);
    i2c.agent.monitor.item_collected_port.connect(freq_adpt_sb.sb_i2c_in);
    freq_generator.agent.monitor.item_collected_port.connect(freq_adpt_sb.sb_osc_gen);
    freq_detector.agent.monitor.item_collected_port.connect(freq_adpt_sb.sb_osc_det);
  endfunction : connect_phase

endclass : freq_adpt_tb

// tb class for SVRNM and VAMS version of frequency adapter
class freq_adpt_ms_tb extends freq_adpt_tb;

  // component macro
  `uvm_component_utils(freq_adpt_ms_tb)
  
  // Constructor
  function new (string name, uvm_component parent=null);
    super.new(name, parent);
  endfunction : new

  // UVM build() phase
  function void build_phase(uvm_phase phase);
    `ifdef UVM_AMS
    // set up bridge proxy pointer references to generator and detector UVCs
    uvm_config_db #(osc_proxy)::set(this,"freq_generator.agent.*","bridge_proxy", top.generator_bridge.proxy);
    uvm_config_db #(osc_proxy)::set(this,"freq_detector.agent.*","bridge_proxy", top.detector_bridge.proxy);
    `endif

    `ifdef DMS_I2C
    uvm_config_db #(i2c_proxy)::set(this,"i2c.agent.*","bridge_proxy", top.i2c_bridge.proxy);
    `endif
    
    // override driver, monitor, and scoreboard with UVM-AMS versions
    `ifdef DMS_I2C
    set_type_override_by_type(i2c_driver::get_type(),i2c_ms_driver::get_type());
    `endif
    set_type_override_by_type(osc_transaction::get_type(),osc_ms_transaction::get_type());
    set_type_override_by_type(osc_driver::get_type(),osc_ms_driver::get_type());
    set_type_override_by_type(osc_monitor::get_type(),osc_ms_monitor::get_type());
    set_type_override_by_type(freq_adpt_scoreboard::get_type(),freq_adpt_ms_scoreboard::get_type());
    
    super.build_phase(phase);

  endfunction 
  
endclass : freq_adpt_ms_tb
